define("addonrock/download-attachments/export-to-zip-download", [
    "addonrock/download-attachments/jszip",
    "addonrock/download-attachments/jszip-utils",
    "addonrock/download-attachments/FileSaver"
], function (JSZip, JSZipUtils, saveAs) {

    const BATCH_SIZE = 8
    const DELAY_BATCH_SIZE_INIT_MS = 0
    const DELAY_BATCH_SIZE_HALF_MS = 15
    const DELAY_BATCH_SIZE_FINAL_MS = 150
    var Promise = window.Promise;
    if (!Promise) {
        Promise = JSZip.external.Promise;
    }

    var DownloadService = function () {
        this.zip = new JSZip();
        this.finishedCounter = 0
        this.expectedCounter = 0
        this.errorCounter = 0
        this.inProgressCounter = 0
        this.filesToDownload = []
        this.zipppingError = false
    };

    function generateFolderNames(attachmentInfos, folderNameCallback) {
        for (var j = 0; j < attachmentInfos.length; j++) {
            attachmentInfos[j].folderName = folderNameCallback(attachmentInfos[j].issue, attachmentInfos[j].attachment);
        }
    }

    function sleep(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    DownloadService.prototype.downloadBatchWithTimeout = function (timeout, fromIndex, toIndex) {
        var thisRef = this;
        sleep(timeout).then(function () {
            thisRef.downloadBatch(fromIndex, toIndex);
            var newFromIndex = Math.min(fromIndex + BATCH_SIZE, thisRef.expectedCounter);
            var newToIndex = Math.min(toIndex + BATCH_SIZE, thisRef.expectedCounter);
            var delayLevel = 0
            if (fromIndex < thisRef.expectedCounter) {
                var delayToUse = DELAY_BATCH_SIZE_INIT_MS;
                if (thisRef.inProgressCounter > BATCH_SIZE) {
                    delayToUse = DELAY_BATCH_SIZE_HALF_MS;
                    delayLevel = 1
                }
                if (thisRef.inProgressCounter > (2 * BATCH_SIZE)) {
                    delayToUse = DELAY_BATCH_SIZE_FINAL_MS;
                    delayLevel = 2
                }
                if (thisRef.inProgressCounter > (3 * BATCH_SIZE)) {
                    delayToUse = 3 * DELAY_BATCH_SIZE_FINAL_MS;
                    delayLevel = 3
                }
                if (thisRef.inProgressCounter > (4 * BATCH_SIZE)) {
                    delayToUse = 10 * DELAY_BATCH_SIZE_FINAL_MS;
                    delayLevel = 4
                }
                var newTimeout = thisRef.inProgressCounter * delayToUse;
                AJS.log("[Download Attachments] Delay for next batch (from, to, newTimeout[ms], delayLevel):", newFromIndex, newToIndex, newTimeout, delayLevel);
                thisRef.downloadBatchWithTimeout(newTimeout, newFromIndex, newToIndex)
            }
        })
    }

    DownloadService.prototype.downloadBatch = function (fromIndex, toIndex) {
        for (var i = fromIndex; i < toIndex; i++) {
            var toDownload = this.filesToDownload[i];
            this.downloadFile(toDownload.url, toDownload.finalName, toDownload.createdDate)
        }
    }

    DownloadService.prototype.addAttachments = function (attachmentInfos, folderNameCallback, messageCallback, percentProgressCallback, finalCallback) {
        this.finalCallback = finalCallback;
        this.percentProgressCallback = percentProgressCallback
        this.messageCallback = messageCallback
        AJS.log("[Download Attachments] Trying to generate zip with attachments count: ", attachmentInfos.length);
        if (attachmentInfos.length <= 0) {
            finalCallback();
        }
        generateFolderNames(attachmentInfos, folderNameCallback);
        attachmentInfos.sort(function (a, b) {
            //AJS.log("sort", new Date(a.attachment.created) - new Date(b.attachment.created));
            return new Date(a.attachment.created) - new Date(b.attachment.created);
        });
        for (var j = 0; j < attachmentInfos.length; j++) {
            //messageCallback("Processing file: " + attachmentInfos[j].attachment.filename);
            var uniqueCounter = 1;
            for (var k = j + 1; k < attachmentInfos.length; k++) {
                if (attachmentInfos[k].folderName === attachmentInfos[j].folderName
                    && attachmentInfos[k].attachment.filename === attachmentInfos[j].attachment.filename) {
                    var oldName = attachmentInfos[k].attachment.filename;
                    var newName = generateUniqueName(oldName, uniqueCounter++);
                    messageCallback("Renaming file '" + oldName + "' to '" + newName + "' - duplicated filename detected.");
                    attachmentInfos[k].attachment.filename = newName;
                }
            }
            var info = attachmentInfos[j];
            this.addFile(info.folderName, info.attachment.filename, info.attachment.created, info.attachment.content);
        }
        this.downloadBatchWithTimeout(0, 0, Math.min(BATCH_SIZE, this.expectedCounter))
    };

    DownloadService.prototype.downloadFile = function (url, finalName, createdDate) {
        const thisRef = this;
        //AJS.log("[Download Attachments] Downloading file: ", url);
        thisRef.inProgressCounter++
        JSZipUtils.getBinaryContent(url, function (err, data) {
            thisRef.finishedCounter++
            thisRef.inProgressCounter--
            thisRef.percentProgressCallback(thisRef.finishedCounter / thisRef.expectedCounter)
            if (err) {
                thisRef.messageCallback("Download FAILED: " + err);
                thisRef.errorCounter++
            } else {
                thisRef.messageCallback("File downloaded: " + finalName);
                thisRef.zip.file(finalName, data, {binary: true, date: new Date(createdDate)}, {});
            }
            if (thisRef.finishedCounter >= thisRef.expectedCounter) {
                thisRef.messageCallback("Downloaded finished ... zipping");
                thisRef.finalCallback()
            }
        });
    };

    DownloadService.prototype.addFile = function (folderName, fileName, createdDate, url) {
        var finalName;
        if (folderName != null) {
            finalName = folderName + fileName;
        } else {
            finalName = fileName;
        }
        //AJS.log("Date: ", createdDate);
        //var localDate = new Date(createdDate).toString().slice(0, 24);
        // AJS.log("local Date: ", localDate);
        // AJS.log("local Date2: ", new Date(localDate));
        // JSZipUtils.getBinaryContent(url, function (err, data) {
        // 	if (err) {
        // 		progressCallback("Download failed: " + err);
        // 	} else {
        // 		console.log("Success")
        // 	}
        // });
        this.expectedCounter++
        this.filesToDownload.push({
            url: url,
            finalName: finalName,
            createdDate: createdDate
        });
        //this.downloadFile(url, finalName, createdDate)
    };

    function generateUniqueName(name, counter) {
        var lastDot = name.lastIndexOf(".");
        var extension = "";
        var filename = "";
        if (lastDot > 0) {
            extension = name.substring(lastDot);
            filename = name.substring(0, lastDot);
        } else {
            filename = name;
        }
        return filename + "." + counter + extension;
        // var firstUnique = ".1";
        // if (name.match(/\.([0-9]+)$/)) {
        // 	var number = parseInt(name.match(/\.([0-9]+)$/)[1], 10) + 1;
        // 	return name.replace(/\.([0-9]+)$/, "." + number);
        // } else {
        // 	return name + firstUnique;
        // }
    }

    function generateDownloadName() {
        var addZero = function (s) {
            if (s < 10)
                return "0" + s;
            else
                return s;
        };
        var baseDownloadName = "attachments_";
        var date = new Date();
        return baseDownloadName + date.getFullYear() + addZero(date.getMonth() + 1) + addZero(date.getDate()) + addZero(date.getHours()) + addZero(date.getMinutes()) + '.zip';
    }

    DownloadService.prototype.download = function (onFinish) {
        //this.messageCallback("Generating zip archive");
        const thisRef = this;
        AJS.log("[Download Attachments] Generating zip archive, saveAs exits?", saveAs.addonrock);
        this.zip.generateAsync({
            type: "blob",
            compression: "STORE"
        }, function (metadata) {
            thisRef.percentProgressCallback(metadata.percent);
        }).then(function (value) {
            const name = generateDownloadName();
            AJS.log("[Download Attachments] Triggering zip save: ", name);
            AJS.log("[Download Attachments] Files count (total / success / error):", thisRef.expectedCounter, thisRef.expectedCounter - thisRef.errorCounter, thisRef.errorCounter);
            thisRef.messageCallback("Files count (total / success / error): (" + thisRef.expectedCounter + " / " + (thisRef.expectedCounter - thisRef.errorCounter) + " / " + thisRef.errorCounter + ")");
            //thisRef.messageCallback("Triggering zip save: " + name);
            try {
                saveAs(value, name);
                AJS.log("[Download Attachments] Save as triggered");
            } catch (e) {
                AJS.log("[Download Attachments] Exception while calling saveAs: ", e);
                thisRef.messageCallback("Save As failed: " + e)
                thisRef.zipppingError = true
            }
            onFinish();
        }, function (reason) {
            AJS.log("[Download Attachments] Failed: " + reason)
            thisRef.messageCallback("Zipping failed: " + reason)
            thisRef.zipppingError = true
        });
    };

    return DownloadService;
});