define("addonrock/download-attachments/export-to-zip-config", [], function () {
	var idGenerator = 0;

	var Rule = function (name, childConstructor) {
		this.name = name;
		this.id = 0;
		this.description = undefined;
		this.custom = {
			component: null
		};
		this.childConstructor = childConstructor !== undefined ? childConstructor : this.constructor;
	};

	Rule.prototype.getLabel = function () {
		return this.description !== undefined ? this.description : this.name;
	};

	Rule.prototype.clone = function (constructor, description) {
		var dest = Object.assign(new constructor(), this);
		var newId = idGenerator++;
		//AJS.log("[Download Attachments] New id generated: ", newId);
		dest.id = newId;
		dest.description = description;
		dest.custom = Object.assign({}, this.custom);
		return dest;
	};

	Rule.prototype.createInstance = function (description) {
		return this.clone(this.childConstructor, description);
	};

	Rule.prototype.apply = function (issue, attachment) {
		//AJS.log("Attachment apply always true", attachment);
		return true;
	};

	function checkFileExtension(fileName, filters) {
		return (new RegExp('(' + filters.join('|').replace(/\./g, '\\.') + "|" + filters.join('|').toUpperCase().replace(/\./g, '\\.') + ')$')).test(fileName);
	}

	var filterAll = new Rule('All (*.*)');

	var WordRule = function () {
		Rule.call(this, 'Word (*.doc, *.docx)', WordRule);
	};
	WordRule.prototype = Object.create(Rule.prototype);
	WordRule.prototype.apply = function (issue, attachment) {
		return checkFileExtension(attachment.filename, ['.doc', '.docx'])
	};

	var PdfRule = function () {
		Rule.call(this, 'PDF (*.pdf)', PdfRule);
	};
	PdfRule.prototype = Object.create(Rule.prototype);
	PdfRule.prototype.apply = function (issue, attachment) {
		return checkFileExtension(attachment.filename, ['.pdf'])
	};
	var ExcelRule = function () {
		Rule.call(this, 'Excel (*.xls, *.xlsx)', ExcelRule);
	};
	ExcelRule.prototype = Object.create(Rule.prototype);
	ExcelRule.prototype.apply = function (issue, attachment) {
		return checkFileExtension(attachment.filename, ['.xls', '.xlsx'])
	};

	var LogRule = function () {
		Rule.call(this, 'Log (*.log)', LogRule);
	};
	LogRule.prototype = Object.create(Rule.prototype);
	LogRule.prototype.apply = function (issue, attachment) {
		return checkFileExtension(attachment.filename, ['.log'])
	};

	var TxtRule = function () {
		Rule.call(this, 'Txt (*.txt)', TxtRule);
	};
	TxtRule.prototype = Object.create(Rule.prototype);
	TxtRule.prototype.apply = function (issue, attachment) {
		return checkFileExtension(attachment.filename, ['.txt'])
	};

	var ImagesRule = function () {
		Rule.call(this, 'Images', ImagesRule);
	};
	ImagesRule.prototype = Object.create(Rule.prototype);
	ImagesRule.prototype.apply = function (issue, attachment) {
		return checkFileExtension(attachment.filename, [
			".ase", ".art", ".bmp", ".blp", ".cd5", ".cit", ".cpt", ".cr2", ".cut", ".dds", ".dib", ".djvu", ".egt", ".exif",
			".gif", ".gpl", ".grf", ".icns", ".ico", ".iff", ".jng", ".jpeg", ".jpg", ".jfif", ".jp2", ".jps", ".lbm", ".max",
			".miff", ".mng", ".msp", ".nitf", ".ota", ".pbm", ".pc1", ".pc2", ".pc3", ".pcf", ".pcx", ".pdn", ".pgm", ".PI1",
			".PI2", ".PI3", ".pict", ".pct", ".pnm", ".pns", ".ppm", ".psb", ".psd", ".pdd", ".psp", ".px", ".pxm", ".pxr", ".qfx", ".raw",
			".rle", ".sct", ".sgi", ".rgb", ".int", ".bw", ".tga", ".tiff", ".tif", ".vtf", ".xbm", ".xcf", ".xpm", ".3dv", ".amf", ".ai",
			".awg", ".cgm", ".cdr", ".cmx", ".dxf", ".e2d", ".egt", ".eps", ".fs", ".gbr", ".odg", ".svg", ".stl", ".vrml", ".x3d", ".sxd",
			".v2d", ".vnd", ".wmf", ".emf", ".art", ".xar", ".png", ".webp", ".jxr", ".hdp", ".wdp", ".cur", ".ecw", ".iff", ".lbm", ".liff",
			".nrrd", ".pam", ".pcx", ".pgf", ".sgi", ".rgb", ".rgba", ".bw", ".int", ".inta", ".sid", ".ras", ".sun", ".tga"
		])
	};

	var CustomRule = function () {
		Rule.call(this, 'Custom expression (?, *)', CustomRule);
		this.custom = {
			value: '*',
			component: {
				props: ['value'],
				model: {
					prop: 'value'
				},
				template: '#download-attachments-add-rule-custom'
			}
		};
	};
	CustomRule.prototype = new Rule('Custom');
	CustomRule.prototype.getLabel = function () {
		return "Custom (" + this.custom.value + ")";
	};

	function testWildcardRegExp(str, rule) {
		return new RegExp("^" + rule.split("*").join(".*").split("?").join(".") + "$").test(str);
	}

	CustomRule.prototype.apply = function (issue, attachment) {
		AJS.log("[Download Attachments] Testing wildcard expression: ", this.custom.value);
		return testWildcardRegExp(attachment.filename, this.custom.value);
	};


	var GroupByIssueKey = function () {
		Rule.call(this, "Issue Key", GroupByIssueKey);
	};
	GroupByIssueKey.prototype = Object.create(Rule.prototype);
	GroupByIssueKey.prototype.apply = function (issue, attachment) {
		return issue.key;
	};

	var GroupByKeyIssueSummary = function () {
		Rule.call(this, "Issue Key&Summary", GroupByKeyIssueSummary);
	};
	GroupByKeyIssueSummary.prototype = Object.create(Rule.prototype);
	GroupByKeyIssueSummary.prototype.apply = function (issue, attachment) {
		return issue.key + " " + issue.fields.summary;
	};

	var GroupByIssueSummary = function () {
		Rule.call(this, "Issue Summary", GroupByIssueSummary);
	};
	GroupByIssueSummary.prototype = Object.create(Rule.prototype);
	GroupByIssueSummary.prototype.apply = function (issue, attachment) {
		return issue.fields.summary;
	};

	var GroupByAttachmentType = function () {
		Rule.call(this, "Attachment Type", GroupByAttachmentType);
	};
	GroupByAttachmentType.prototype = Object.create(Rule.prototype);
	GroupByAttachmentType.prototype.apply = function (issue, attachment) {
		return getFileExtension(attachment.filename);
	};

	function getFileExtension(filename) {
		var split = filename.split('.');
		return split.length > 1 || filename.indexOf(".") > -1 ? split.pop() : "";
	}

	var groupByIssueKey = new GroupByIssueKey();
	var groupByIssueSummary = new GroupByIssueSummary();
	var groupByKeyIssueSummary = new GroupByKeyIssueSummary();
	var groupByAttachmentType = new GroupByAttachmentType();

	var dictionaries;

	var configFactory = {};
	configFactory.getDictionaries = function () {
		dictionaries = dictionaries || {
			filterRules: [
				filterAll,
				new WordRule(),
				new PdfRule(),
				new ExcelRule(),
				new TxtRule(),
				new LogRule(),
				new ImagesRule(),
				new CustomRule()
			],
			grouping: [
				groupByIssueKey,
				groupByIssueSummary,
				groupByAttachmentType,
				groupByKeyIssueSummary
			]
		};
		return dictionaries;
	};

	configFactory.createDefaultFilterRules = function () {
		return [
			{
				title: 'Include if matches',
				color: 'green',
				multiple: true,
				rules: [
					filterAll.createInstance()
				]
			},
			{
				title: 'Exclude if matches', color: 'red',
				rules: []
			}
		];
	};


	configFactory.createDefaultGroupingRules = function () {
		return [
			{
				title: 'Group by',
				color: 'green',
				multiple: false,
				rules: [
					groupByIssueKey.createInstance()
				]
			}
		];
	};

	return configFactory;
});