define('tmatesoft/svnmirror/util/poll', [
    'jquery',
    'exports'
], function(
    $,
    exports) {

    var DONE = (typeof XMLHttpRequest.DONE !== 'undefined') ? XMLHttpRequest.DONE : 4;
    var LOADING = (typeof XMLHttpRequest.LOADING !== 'undefined') ? XMLHttpRequest.LOADING : 3;
    var HEADERS_RECEIVED = (typeof XMLHttpRequest.HEADERS_RECEIVED !== 'undefined') ? XMLHttpRequest.HEADERS_RECEIVED : 2;

    var supportsLongPolling = function() {
        return !($.browser.msie && parseInt($.browser.version, 0) < 10);
    };

    var isComplete = function(xhr) {
        return xhr && xhr.readyState == DONE;
    };

    var isCompleteWithError = function(xhr) {
        return isStatusAvailable(xhr) && xhr.status > 200;
    };

    var isTextAvailable = function(xhr) {
        return xhr && xhr.readyState >= LOADING;
    };

    var isStatusAvailable = function(xhr) {
        return xhr && xhr.readyState >= HEADERS_RECEIVED;
    };

    exports.poll = function(url, shortPollUrl, pollOptions) {
        var messageSeparator = '\r\n';
        var messageSeparatorLength = messageSeparator.length;

        var self = {};

        self.serverPollTimeout = pollOptions.interval * 1000;
        self.pollingEnabled = pollOptions.enabled;

        var maxPollInterval = self.pollingEnabled ? 1000 : 5000;
        var minPollInterval = self.pollingEnabled ? 1000 : 2000;
        var pollIntervalIncrement = 200;
        var shortPollFallbackTimeout = 10000;
        var shortPollTimeout = 15000;

        if (!supportsLongPolling()) {
            self.serverPollTimeout = 3000;
            self.pollingEnabled = false;
        }

        self.pollInterval = self.pollingEnabled ? pollIntervalIncrement : minPollInterval;
        self.dataRead = false;

        self.data = function(receiver) {
            self.receiver = receiver;
            return self;
        };

        self.error = function(receiver) {
            self.errorReceiver = receiver;
            return self;
        };

        self.log = function(receiver) {
            self.logReceiver = receiver;
            return self;
        };

        window.onbeforeunload = function() {
            self.errorState = true;
        };

        self.poll = function() {
            self.doPoll(self);
        };

        self.doPoll = function(self) {
            if (self.errorState) {
                return;
            }
            delete self.nextScheduled;

            if (self.xhr && self.pollingEnabled) {
                if (isCompleteWithError(self.xhr)) {
                    self.reportError();
                    return;
                }
                if (isTextAvailable(self.xhr)) {
                    try {
                        self.readData(self.xhr);
                    } catch (err) {
                        self.reportError(err);
                        return;
                    }
                }
                if (self.errorState) {
                    return;
                }
                if (isComplete(self.xhr)) {
                    self.xhr = null;
                } else {
                    if (!self.dataRead && (((new Date().getTime()) - self.startTime) > shortPollFallbackTimeout)) {
                        self.logMessage('No data was read in ' + shortPollFallbackTimeout + ' ms.');
                        self.logMessage('Falling back to short polling algorithm.');

                        if (self.xhr) {
                            self.errorState = true;
                            self.xhr.abort();
                            delete self.errorState;
                            self.xhr = null;
                        }

                        self.pollingEnabled = false;
                        self.serverPollTimeout = 3000;
                        maxPollInterval = 5000;
                        minPollInterval = 2000;
                        self.pollInterval = minPollInterval;
                        self.nextScheduled = window.setTimeout(self.doPoll.bind(this, self), self.pollIntervalIncrement);
                    }
                }
            } else {
                self.xhr = new XMLHttpRequest();
                self.xhr.onerror = function(event) {
                    if (self.errorState) {
                        return;
                    }
                    self.logMessage('Error on sending XML HTTP request');
                    self.retry(true);
                };

                self.xhr.onreadystatechange = function(event) {
                    self.logMessage('XHR.onReadyStateChange(' + event + '), errorState: ' + self.errorState);
                    if (self.errorState) {
                        return;
                    }

                    if (self.xhr) {
                        self.logMessage('XHR.status: ' + self.xhr.status);
                        self.logMessage('XHR.readyState: ' + self.xhr.readyState);
                    } else {
                        self.logMessage('XHR: ' + self.xhr);
                    }
                    self.logMessage('isCompleteWithError: ' + isCompleteWithError(self.xhr));
                    self.logMessage('isComplete: ' + isComplete(self.xhr));
                    self.logMessage('isTextAvailable: ' + isTextAvailable(self.xhr));

                    if (isCompleteWithError(self.xhr)) {
                        self.logMessage('XHR is complete, but with an error.');
                        self.reportError();
                    } else if (isComplete(self.xhr)) {
                        self.logMessage('XHR is complete, pollingEnabled: ' + self.pollingEnabled);
                        if (self.pollingEnabled) {
                            self.now();
                        } else {
                            self.logMessage('About to read data: ' + isTextAvailable(self.xhr));
                            if (isTextAvailable(self.xhr)) {
                                try {
                                    self.readData(self.xhr);
                                    self.logMessage('data read');
                                } catch (err) {
                                    self.logMessage('error on readData(xhr) call: ' + err.message);
                                    self.logMessage(err.stack);
                                    self.reportError(err);
                                }
                            }
                            self.xhr = null;
                            self.logMessage('XHR disposed, scheduling next one');
                            self.nextScheduled = window.setTimeout(self.doPoll.bind(this, self), self.pollInterval);
                            self.logMessage('next XHR scheduled');
                        }
                    }
                };
                self.lastRead = 0;
                try {
                    if (self.pollingEnabled) {
                        self.logMessage('Establishing background long poll connection' +
                            ' (will try for ' + (shortPollFallbackTimeout / 1000) + ' seconds)...');
                    } else {
                        self.logMessage('Fetching data with GET request' +
                            ' (will try for ' + (shortPollTimeout / 1000) + ' seconds)...');
                        self.xhr.ontimeout = function(e) {
                            self.logMessage('No data was read in ' + shortPollTimeout + ' ms.');
                            self.logMessage('Giving up.');
                            self.reportError({ message : 'Timeout reading mirror data.' });
                        };
                    }
                    var requestMethod = self.pollingEnabled ? 'POST' : 'GET';
                    var requestUrl = self.pollingEnabled ? url : shortPollUrl;
                    self.xhr.open(requestMethod, requestUrl, true);
                    if (!self.pollingEnabled) {
                        self.xhr.timeout = shortPollTimeout;
                    }
                    self.xhr.setRequestHeader('Content-type', 'application/json; charset=utf-8');
                    self.xhr.setRequestHeader('Cache-Control', 'no-cache');
                    self.xhr.setRequestHeader('Cache-Control', 'no-store');
                    self.xhr.setRequestHeader('Pragma', 'no-cache');
                    self.startTime = new Date().getTime();
                    self.xhr.send();
                    self.dataRead = false;
                } catch (err) {
                    self.logMessage('Error on sending XML HTTP Request: ' + err.message);
                    self.retry(false, err);
                    return;
                }
            }

            if (self.pollingEnabled) {
                var pollTime = new Date().getTime() - self.startTime;
                if (pollTime + self.pollInterval >= self.serverPollTimeout) {
                    self.pollInterval = Math.max(100, self.serverPollTimeout - pollTime + 100);
                }
                self.nextScheduled = window.setTimeout(self.doPoll.bind(this, self), self.pollInterval);
            }
        };

        self.resetRetry = function() {
            self.retryAttemptsLeft = 3;
            self.retryLog = '';
            self.retryInterval = 2500;
        };

        self.now = function() {
            if (self.errorState) {
                return;
            }
            while(!self.nextScheduled) {}

            window.clearTimeout(self.nextScheduled);
            delete self.nextScheduled;

            self.pollInterval = 0;
            self.poll();
        };

        self.stop = function() {
            self.errorState = true;
            if (self.nextScheduled) {
                window.clearTimeout(self.nextScheduled);
                delete self.nextScheduled;
            }
            self.data(null);
            self.error(null);
            self.log(null);

            if (self.xhr) {
                self.xhr.abort();
                self.xhr = null;
            }
        };

        self.logMessage = function(message) {
            if (self.errorState) {
                return;
            }
            self.retryLog += message;
            self.retryLog += '\n';

            if (self.logReceiver) {
                self.logReceiver(message);
            }
        }

        self.readData = function(xhr) {
            if (self.errorState) {
                return;
            }
            self.logMessage('typeof(xhr.responseText): ' + typeof(xhr.responseText));
            if (typeof(xhr.responseText) == 'unknown') {
                self.pollInterval = Math.min(maxPollInterval, self.pollInterval + pollIntervalIncrement);
                return;
            }
            var unprocessed = xhr.responseText;
            if (!self.dataRead) {
                self.dataRead = unprocessed && unprocessed.length > 0;
            }
            self.logMessage('unprocessed.length: ' + unprocessed.length);

            var messageEnd = unprocessed.indexOf(messageSeparator, self.lastRead);
            self.logMessage('message end index: ' + messageEnd);
            if (messageEnd < 0) {
                self.pollInterval = Math.min(maxPollInterval, self.pollInterval + pollIntervalIncrement);
                return;
            }
            if (self.pollingEnabled) {
                self.pollInterval = minPollInterval;
            }
            self.resetRetry();
            while(messageEnd > 0) {
                var messageStart = self.lastRead;
                var message = unprocessed.substring(self.lastRead, messageEnd);
                if (!self.pollingEnabled && self.lastMessage === message) {
                    self.logMessage('message is the same as the last one, skipping');
                    self.pollInterval = Math.min(maxPollInterval, self.pollInterval + pollIntervalIncrement);
                } else {
                    self.pollInterval = minPollInterval;
                    if (message !== '') {
                        self.logMessage('parsing message');
                        var parsed = JSON.parse(message);
                        self.logMessage('parsed: ' + parsed);
                        if (parsed) {
                            self.receiver(parsed, self.lastRead == 0);
                        }
                    }
                    if (!self.pollingEnabled) {
                        self.lastMessage = message;
                        self.logMessage('last message saved');
                    }
                }
                self.lastRead = messageEnd + messageSeparatorLength;
                messageEnd = unprocessed.indexOf(messageSeparator, self.lastRead);
                self.logMessage('next message end index: ' + messageEnd);
            }
        };

        self.retry = function(unschedule, err) {
            if (self.retryAttemptsLeft == 0 || !self.pollingEnabled) {
                self.reportError(err);
            } else {
                if (unschedule) {
                    while(!self.nextScheduled) {}
                    window.clearTimeout(self.nextScheduled);
                    delete self.nextScheduled;
                }
                self.retryAttemptsLeft--;
                self.logMessage('Will retry to establish background connection in ' + self.retryInterval + ' ms.');
                window.setTimeout(self.doPoll.bind(this, self), self.retryInterval);
            }
        };

        self.reportError = function(err) {
            if (self.errorState) {
                return;
            }
            self.errorState = true;
            var msg;
            if (err) {
                msg = err.message || err.statusText;
                if (msg && err.stack) {
                    msg += '\n' + err.stack;
                }
            } else {
                msg = isStatusAvailable(self.xhr) ? self.xhr.statusText : null;
            }
            msg = (msg || 'Connection Lost') + (self.retryLog !== '' ? '\n\nLast log:\n' + self.retryLog : '');
            if (self.errorReceiver) {
                self.errorReceiver(msg);
            }

            self.stop();
        };

        self.resetRetry();

        return self;
    };
});