define('tmatesoft/svnmirror/controller', [
    'jquery',
    'exports',
    'tmatesoft/svnmirror/util/form'
    ],
function(
    $,
    exports,
    form
) {

    var isArray = function(o) {
        if (!Array.isArray) {
            return Object.prototype.toString.call(o) === '[object Array]';
        }
        return Array.isArray(o);
    }

    var isObject = function(o) {
        return o && o.constructor == Object;
    }

    var isString = function(o) {
        return typeof o == 'string' || o instanceof String;
    }

    var modelProperties = function(model, cb, id) {
        id = id || '';
        var count = 0;
        for(var p in model) {
            if (!model.hasOwnProperty(p)) {
                continue;
            }
            var valId = id == '' ? p : id + '.' + p;
            if (isArray(model[p])) {
                cb(valId, model[p])
            } else if (isObject(model[p])) {
                modelProperties(model[p], cb, valId);
            } else {
                cb(valId, model[p])
            }
            count++;
        }
        if (count == 0) {
            cb(id, model);
        }
    };

    var setModelValue = function(object, id, value) {
        if (isString(value)) {
            value = value.replace('\r\n', '\n');
            value = value.replace('\r', '\n');
            value = value.replace('\t', '    ');
        }

        var segments = id.split('.');
        for(var i = 0; i < segments.length; i++) {
            if (i == segments.length - 1) {
                object[segments[i]] = value;
            } else {
                if (!object[segments[i]]) {
                    object[segments[i]] = {};
                }
                object = object[segments[i]];
            }
        }
    };

    var getModelValue = function(object, id) {
        var segments = id.split('.');
        for(var i = 0; i < segments.length; i++) {
            if (i == segments.length - 1) {
                return object[segments[i]];
            } else {
                if (!object[segments[i]]) {
                    return null;
                }
                object = object[segments[i]];
            }
        }
        return null;
    };



    var actionElements = function(parent, cb) {
        parent.find('[id$=-button]').each(function() {
            var action = $(this).attr('id');
            action = action.substring(0, action.indexOf('-'));
            cb($(this), action);
        });
    }

    exports.view = function(templateFactory) {

        var self = {
            render : function(selector, state, actions) {
                self.state = state;
                var template = templateFactory(self.state);
                self.view = form.renderTemplate(selector, template);
                actionElements(self.view, function(el, action) {
                    if (typeof actions[action] === 'function') {
                        el.click(actions[action]);
                    }
                });

                self.tweak();
                self.bind();
                self.update();
            },

            modelUpdated : function(delta) {
                modelProperties(delta, function(id, value) {
                    var remoteValue = getModelValue(self.remoteState, id);
                    var localValue = getModelValue(self.state, id);
                    if (remoteValue != localValue) {
                        return;
                    }
                    var uiElement = self.getViewElement(id);
                    if (uiElement && uiElement.length > 0) {
                        form.setUIValue(uiElement, value);
                    }
                });
                self.update();
            },

            revert : function(state) {
                self.state = state;
                modelProperties(self.state, function(id, value) {
                    var uiElement = self.getViewElement(id);
                    if (uiElement && uiElement.length > 0) {
                        form.setUIValue(uiElement, value);
                    }
                });
                self.update();
            },

            disable : function() {
                if (self.view) {
                    actionElements(self.view, function(el, action) {
                        el.attr('disabled', 'disabled');
                    });
                }
            },

            dispose : function() {
                self.state = null;
                self.view = null;
            },

            bind : function() {},
            tweak : function() {},
            update : function() {},

            getViewElement : function(modelId) {
                return null;
            },
        };
        self.state = {};
        return self;
    };

    exports.setModelValue = setModelValue;
    exports.getModelValue = getModelValue;

    exports.create = function(statusSelector, formSelector) {
        var self = {
            update : function(delta) {
                self.updateModel(delta);
                if (self.currentView) {
                    self.currentView.remoteState = self.remoteState;
                }

                var viewKey = self.state.data ? self.state.data.stage : '';
                var newView = self.views[viewKey];
                if (newView && newView !== self.currentView) {
                    if (self.currentView) {
                        self.currentView.dispose();
                    }
                    self.state = JSON.parse(JSON.stringify(self.remoteState));
                    self.currentView = newView;
                    self.currentView.remoteState = self.remoteState;
                    self.currentView.render(formSelector, self.state, self.actions);
                } else if (newView && newView === self.currentView) {
                    self.currentView.modelUpdated(delta);
                }

                self.statusView.render(statusSelector, self.state, self.actions);
            },

            updateModel : function(delta) {
                modelProperties(delta, function(id, value) {
                    var oldValue = getModelValue(self.remoteState, id);
                    if (isObject(value) || isObject(oldValue)) {
                        setModelValue(self.state, id, value);
                    } else if (isArray(value) || isArray(oldValue)) {
                        setModelValue(self.state, id, value);
                    } else {
                        var uiValue = getModelValue(self.state, id);
                        if (oldValue == uiValue && value != oldValue) {
                            setModelValue(self.state, id, value);
                        }
                    }
                    setModelValue(self.remoteState, id, value);
                });
            },

            disable : function() {
                for(var name in self.views) {
                    if (self.views.hasOwnProperty(name)) {
                        self.views[name].disable();
                    }
                }
                self.statusView.disable();
                return self;
            },

            revert : function() {
                self.state = JSON.parse(JSON.stringify(self.remoteState));
                if (self.currentView) {
                    self.currentView.revert(self.state);
                }
            },

            on : function(name, action) {
                self.actions[name] = function(event) {
                    event.preventDefault();
                    action(name, self.state);
                    return false;
                };
                return self;
            },

            view : function(name, view) {
                if (!view) {
                    return self.views[name];
                }
                self.views[name] = view;
                view.controller = self;
                return self;
            },

            status : function(view) {
                self.statusView = view;
                view.controller = self;
                return self;
            },

            hasChanges : function() {
                if (!self.state.data || !self.remoteState.data) {
                    return true;
                }
                return JSON.stringify(self.state.data) != JSON.stringify(self.remoteState.data);
            }
        };

        self.state = {};
        self.remoteState = {};
        self.actions = {};
        self.views = {};

        return self;
    };
});