define('page/plugin/svn-import-authors',
['exports', 'jquery', '@atlassian/aui'],
function(exports, $, AJS) {

    function Dialog(params) {
        this.id = params.id;
        this.title = params.title;
        this.modified = false;
        this.comparator = params.comparator;
        this.validators = params.validators;
        this.canAdd = false;
        this.errorMessage = null;
        this.canSubmit = true;
    }

    Dialog.prototype.setAuthorsPanel = function(p) {
        this.panel = p;
        var self = this;
        $('.revert', this.panel).bind('click', function() {
            self.revert();
        });
        $('.apply', this.panel).bind('click', function() {
            self.apply(true);
        });
        $('.load', this.panel).bind('click', function() {
            self.loadAuthors();
        });
        $('#authorsFileChooser', this.panel).change(function(event) {
            var file = $(this).val();
            self.uploadAuthorsFile(event.target.files[0])
            $(this).val("");
        });
    }

    Dialog.prototype.uploadAuthorsFile = function(file) {
        var self = this;
        var xhr = new XMLHttpRequest();

        xhr.open('POST', document.URL + '/parseAuthors');
        xhr.responseType='json';
        xhr.onreadystatechange = function(e) {
            if (this.readyState == 4) {
                if (this.status == 200) {
                    self.showAuthorsPopupAndLoad(this.response);
                }  else {
                    alert('Error parsing authors file:\n' + this.status + " " + this.statusText);
                }
            }
        };
        xhr.send(file);
    }

    Dialog.prototype.showAuthorsPopupAndLoad = function(loadedData) {
        var self = this;
        var popup = new AJS.Dialog({
            width : 600,
            id : 'authors-dialog',
            closeOnOutsideClick: false
        });
        popup.addHeader('Load Authors Mapping');
        if (loadedData.length > 0) {
            var records = "<div style='height: 250px; overflow-y: scroll;'><table ><tbody>";
            for(var i in loadedData) {
                records += "<tr><td width=\"150\">";
                records += loadedData[i].svnName + " =&gt;</td><td width=\"350\">";
                records += loadedData[i].gitName + "&nbsp;<b>&lt;";
                records += loadedData[i].gitEmail + "&gt;</b></td></tr>";
            }
            records += "</tbody></table></div>";
            if (loadedData.length > 1) {
                popup.addPanel('Authors', '<p>There are ' + loadedData.length + ' author mapping records found:<p/>'
                                    + records
                                    + '<p>How would you like to load these records?</p>');
            } else {
                popup.addPanel('Authors', '<p>There is one author mappings record found:<p/>'
                                    + records
                                    + '<p>How would you like to load that record?</p>');
            }
            popup.addSubmit('Replace If Already Exists', function() {
                self.load(self.updateMappings(self.data, loadedData, true));
                popup.hide();
                popup.remove();
            });
            popup.addButton('Add New Only', function() {
                self.load(self.updateMappings(self.data, loadedData, false));
                popup.hide();
                popup.remove();
            })
            popup.addButton('Replace All', function() {
                self.load(loadedData);
                popup.hide();
                popup.remove();
            });
        } else {
            popup.addPanel('Authors', '<p>No authors mapping has been found.<br/>'
                                    + 'Git-svn authors.txt file <a target="_blank" href=\"http://subgit.com/book-remote/index.html#authors-mapping\">format</a> is expected.</p>');
        }
        popup.addLink('Cancel', function (dialog) { dialog.hide(); dialog.remove(); })
        popup.show();
    }

    Dialog.prototype.updateMappings = function(oldData, newData, replaceExisting) {
        var update = oldData.slice(0);
        for(var i in newData) {
            var found = false;
            for(var j in update) {
                if (update[j].svnName == newData[i].svnName) {
                    if (replaceExisting) {
                        update[j] = newData[i];
                    }
                    found = true;
                    break;
                }
            }
            if (!found) {
                update.push(newData[i]);
            }
        }
        return update;
    }

    Dialog.prototype.validateField = function(field) {
        if (this.validators != null) {
            var validator = this.validators[field.attr('name')];
            if (typeof validator == 'function') {
                return validator(field.val());
            }
        }
        return null;
    }

    Dialog.prototype.load = function(data) {
        this.data = data;
        this.revert();
    }

    Dialog.prototype.apply = function(callCallback) {
        var newData = [];
        $('.authors-table-body .author-row', this.panel).each(function() {
            var author = buildAuthor($(this));
            newData.push(author);

            $(this).find('input').each(function() {
                $(this).attr('defaultValue', author[$(this).attr('name')]);
            });
        });
        this.data = newData;
        this.modified = false;

        if (callCallback && typeof this['onApply'] == 'function') {
            this['onApply']();
            adjustButtons(this);
        }
    }

    Dialog.prototype.revert = function() {
        $('.authors-table-body', this.panel).empty();
        // remove all listeners?
        for(var i = 0; i < this.data.length; i++) {
            var author = this.data[i];
            var row = buildRow(author);

            $('.authors-table-body', this.panel).append(row);
            this.rowAdded(author);
        }
        this.modified = false;
        this.reveal(0);
        adjustHeight(this);
        adjustWidth(this);
        adjustButtons(this);
        clearInputs(this);
        setupInputValidators(this);
    }

    Dialog.prototype.loadAuthors = function() {
        $('#authorsFileChooser').trigger('click')
    }

    function setupInputValidators(dialog) {
        $('.authors-table-header input', this.panel).unbind('keyup change focusin');
        $('.authors-table-header input', this.panel).bind('keyup change focusin', function(event) {
            dialog.errorMessage = null;
            dialog.canAdd = dialog.validateField($(this)) == null;
            if (dialog.canAdd) {
                $('.authors-table-header input', this.panel).each(function () {
                    dialog.canAdd &= (dialog.validateField($(this)) == null);
                    if (!dialog.canAdd) {
                        return false;
                    }
                });
            }
            adjustButtons(dialog);
        });
    }

    Dialog.prototype.insert = function(author) {
        var self = this;
        var selector = $('.authors-table-body .author-row', this.panel);
        var rowCount = selector.length;
        var addedRowIndex = -1;
        selector.each(function(index) {
            var rowAuthor = buildAuthor($(this));
            if (self.comparator(author, rowAuthor) <= 0) {
                $(buildRow(author)).insertBefore($(this));
                addedRowIndex = index;
                self.rowAdded(author);
                return false;
            }
        });
        if (addedRowIndex < 0) {
            $('.authors-table-body', this.panel).append($(buildRow(author)));
            addedRowIndex = rowCount;
            self.rowAdded(author);
        }
        this.modified = true;
        adjustWidth(this);
        adjustButtons(this);

        return addedRowIndex;
    }

    Dialog.prototype.reveal = function(index) {
        var bodyHeight = $('.authors-table-body', this.panel).height();
        var totalRowCount = $('.authors-table-body .author-row', this.panel).length;
        var rowHeight = totalRowCount > 0 ? $('.authors-table-body .author-row', this.panel).outerHeight(true) : 0;
        var rowsShown = bodyHeight / rowHeight;
        //
        if (totalRowCount <= rowsShown) {
            index = 0;
        } else if (totalRowCount - index < rowsShown) {
            index = totalRowCount - rowsShown;
        }
        $('.authors-table-body', this.panel).scrollTop(index * rowHeight);
    }

    exports.dialog = Dialog;

    var fieldListener = {
        onFieldFocusIn : function(event) {
            var self = event.data['dialog']
            $(this).one('mouseup', function(e){
                e.preventDefault();
            });
            $(this).select();

            originalValue = $(this).val();
            $(this).bind('keyup onchange', function(event) {
                self.errorMessage = self.validateField($(this));
                self.canSubmit = self.errorMessage == null;

                var defaultValue = $(this).attr('defaultValue');
                if (defaultValue != $(this).val()) {
                    if (!self.modified) {
                        self.modified = true;
                    }
                }
                adjustButtons(self);
            });
            self.errorMessage = null;
            adjustButtons(self);
        },

        onFieldFocusOut : function(event) {
            var self = event.data.dialog;
            $(this).unbind('keyup onchage');

            if (self.validateField($(this)) != null) {
                $(this).val(originalValue);
                self.canSubmit = true;
            }
            self.errorMessage = null;
            adjustButtons(self);
        },

        onDelete : function(event) {
            var self = event.data.dialog;
            event.preventDefault();
            $(this).parent().parent().remove();
            self.modified = true;

            adjustWidth(self);
            adjustButtons(self);

            return false;
        },
    }

    Dialog.prototype.rowAdded = function(author) {
        var self = this;
        $('#added a.delete-mapping').one('click', {dialog : this}, fieldListener.onDelete);

        $('#added input').each(function() {
            if ($(this).attr('disabled') == null) {
                $(this).bind('focusin', {dialog : self}, fieldListener.onFieldFocusIn);
                $(this).bind('focusout', {dialog : self}, fieldListener.onFieldFocusOut);

            }
        })
        $('#added').removeAttr('id');
    }

    function adjustHeight(dialog) {
        var dialogHeight = $('#authors-tabs').height() - 60;
        var headerHeight = $('.authors-table-header', dialog.panel).outerHeight(true);
        var footerHeight = $('.authors-table-buttons', dialog.panel).outerHeight(true);
        var targetHeight = dialogHeight - headerHeight - footerHeight;
        $('.authors-table-body', dialog.panel).css('height', targetHeight);
        $('.authors-table-body', dialog.panel).css('max-height', targetHeight);
    }

    function adjustWidth(dialog) {
        var height = $('.authors-table-body', dialog.panel).height();
        var rowCount = $('.authors-table-body .author-row', dialog.panel).length;
        var rowsHeight = rowCount * $('.authors-table-body .author-row', dialog.panel).outerHeight(true);

        var hasScrollbar = rowsHeight > height;

        var margin = hasScrollbar ? '3px' : '20px';
        $('.authors-table-body', dialog.panel).css('margin-right', margin);
    }

    function adjustButtons(dialog) {
        enable('.authors-table-buttons .apply', dialog.canSubmit && dialog.modified);
        enable('.authors-table-buttons .revert', dialog.modified);
        enable('.authors-table-buttons .load', true);
        enable('#authorsFileChooser', true);


        enable('.authors-table-header .add-mapping', dialog.canAdd);
        enable(dialog.submitSelector, dialog.canSubmit);

        if (dialog.errorMessage != null) {
            $('.authors-table-buttons .authors-message', dialog.panel).css('color', 'red');
            $('.authors-table-buttons .authors-message', dialog.panel).html(dialog.errorMessage);
        } else {
            $('.authors-table-buttons .authors-message', dialog.panel).css('color', '');
            $('.authors-table-buttons .authors-message', dialog.panel).html('Load mapping from file (git-svn authors.txt <a target="_blank" href="http://subgit.com/book-remote/index.html#authors-mapping">format<a/>)&nbsp;:');
        }
    }

    function enable(selector, enabled) {
        if (enabled) {
            $(selector).removeAttr('disabled');
        } else {
            $(selector).attr('disabled', 'true');
        }

    }

    function clearInputs(dialog) {
        $('.authors-table-header input', dialog.panel).val('');
        $('.authors-table-header input[autofocus=true]', dialog.panel).focus();
    }

    function buildRow(authorData) {
        return plugin.page.authorRow({author: authorData});
    }

    function buildAuthor(row) {
        var author = {};
        row.find('input').each(function() {
            author[$(this).attr('name')] = $(this).val();
        });
        return author;
    }
});
