define('tmatesoft/svn/views/status-view', [
    '@atlassian/aui',
    'tmatesoft/svn/validation',
    'tmatesoft/svn/messages',
    'tmatesoft/svn/views/status-util'
], function(
    AJS,
    Validation,
    Messages,
    StatusUtil
) {
    'use strict';

    function StatusView(commandTracker) {
        this.commandTracker = commandTracker;
    };

    StatusView.prototype.render = function(model, el, $) {
        el.innerHTML = tmatesoft.svn.status({ stage : model.get('data.stage') });
    };

    StatusView.prototype.setup = function(model, el, $) {
        var self =this;
        var item = model.get();
        if (item.task) {
            StatusUtil.updatePrimaryBadge(item, $);
            StatusUtil.updateSecondaryBadge(item, $);
            StatusUtil.updateMessage(item, $);
        }
        this.commandListener = this.commandTracker.on('sent completed cancelled', model.get('scope.id'), function() {
            self.updateButtonsState(model, $);
        });
        $('#command, #uninstall').click(function(e) {
            var command = $(this).attr('data-command');
            self.commandTracker.sendCommand(command, model.get());
        });
        $('#unsynced-link').click(function(e) {
            e.preventDefault();
            AJS.tabs.change($.$('a[href="#unsynced-tab"]'));
            $.$('a[href="#unsynced-tab"]').focus();
        });
        this.updateButtonsText(model, $);
        this.updateUnsyncedNotification(model.get('data'), $);
        this.updateErrorNotification(model.get('task'), $);
        this.updateSvnLink(model.get(), $);
    };

    StatusView.prototype.update = function(event, el, $) {
        if (event.changed('data.stage')) {
            return true;
        }
        if (event.changed('task')
            || event.changed('pending')
            || event.changed('data.hasUnsyncedRefs')) {

            StatusUtil.updatePrimaryBadge(event.get(), $);
            StatusUtil.updateSecondaryBadge(event.get(), $);
            StatusUtil.updateMessage(event.get(), $);

            this.updateUnsyncedNotification(event.get('data'), $);
            this.updateErrorNotification(event.get('task'), $);
            this.updateButtonsText(event, $);
        }
        if (event.changed('data.url')) {
            this.updateSvnLink(event.get(), $);
        }
    };

    StatusView.prototype.dispose = function(event, el, $) {
        this.commandTracker.off(this.commandListener);
        this.commandListener = null;
    };

    StatusView.prototype.updateButtonsState = function(model, $) {
        $('#command, #uninstall').prop('disabled', this.commandTracker.isCommandSubmitted(model.get('scope.id')));
    };

    StatusView.prototype.updateButtonsText = function(model, $) {
        var command = StatusUtil.getCommandForStage(model.get());

        $('#command').text(command.label).attr('data-command', command.name);
        $('#command').prop('disabled', command.name === 'unknown');
        $('#command').attr('title', command.title);
        var stage = model.get('data.stage');
        if (stage === 'INITIAL' || stage === 'CONFIGURED') {
            $('#command').toggle(command.name !== 'unknown');
        }
    };

    StatusView.prototype.updateUnsyncedNotification = function(data, $) {
        if (data.stage === 'INSTALLED' || data.stage === 'PAUSED') {
            $('#unsynced-notification').toggle(data.hasUnsyncedRefs);
        } else {
            $('#unsynced-notification').toggle(false);
        }
    };

    StatusView.prototype.updateErrorNotification = function(task, $) {
        if (task.state === 'FAILED') {
            var error = task.message || 'unknown error';
            if (task.name === 'push' || task.name === 'sync' || task.name === 'upgrade') {
                var prefix = task.name === 'upgrade' ? 'Upgrade failed: ' : 'Last synchronization failed with an error: ';
                var errorElement = $('<p></p>').text(prefix).append($('<code></code>').text(error));
                $('#error-notification').empty().append(errorElement);
                $('#error-notification').toggle(true);
                return;
            }
        }
        $('#error-notification').toggle(false);
    };

    StatusView.prototype.updateSvnLink = function(model, $) {
        var url = model.data.url;
        $('.svn-link').toggle(url !== null);
        $('.svn-link-text').toggle(url !== null);
        $('.svn-link-icon').toggle(url !== null);
        if (url !== null) {
            if (url.indexOf('http') >= 0) {
                $('.svn-link').attr('href', url);
                $('.svn-link').text(url);
                $('.svn-link').toggle(true);
                $('.svn-link-text').toggle(false);
            } else {
                $('.svn-link-text').text(url);
                $('.svn-link').toggle(false);
                $('.svn-link-text').toggle(true);
            }
            if (model.data.importRequested) {
                if (model.data.stage === 'INSTALLED') {
                    $('.svn-link-icon').attr('title', 'Importing from Subversion project at\n' + url);
                } else {
                    $('.svn-link-icon').attr('title', 'Imported from Subversion project at\n' + url);
                }
            } else {
                $('.svn-link-icon').attr('title', 'Mirroring Subversion project at\n' + url);
            }
        }
    };

    return StatusView;
});