define('tmatesoft/svn/views/settings/credentials-settings', [
    'tmatesoft/svn/validation',
    'tmatesoft/svn/views/util'
], function(
    Validation,
    Util
) {

    'use strict';

    function CredentialsSettings(commandTracker) {
        this.commandTracker = commandTracker;
    };

    CredentialsSettings.prototype.render = function(model, el, $) {
        var html = '';
        html += tmatesoft.svn.settings.credentials();
        el.innerHTML = html;
    };

    CredentialsSettings.prototype.setup = function(model, el, $) {
        var data = model.get('data');
        var url = model.get('data.url');
        var scopeId = model.get('scope.id');

        this.updateFields(data, el, $);
        this.updateTestButton(url, data.credentials, el, $);

        Util.trackFieldChanges($('.credentials-type'), $, function(val) { model.set('data.credentials.type', val, this); });
        Util.trackFieldChanges($('#useCache'), $, function(val) { model.set('data.credentials.useCache', val, this); });
        Util.trackFieldChanges($('#userName'), $, function(val) { model.set('data.credentials.userName', val, this); });
        Util.trackFieldChanges($('#password'), $, function(val) { model.set('data.credentials.password', val, this); });
        Util.trackFieldChanges($('#sslPassword'), $, function(val) { model.set('data.credentials.sslPassword', val, this); });
        Util.trackFieldChanges($('#passphrase'), $, function(val) { model.set('data.credentials.passphrase', val, this); });
        Util.trackFieldChanges($('#useSshAgent'), $, function(val) { model.set('data.credentials.useSshAgent', val, this); });

        // file upload.
        $('#sshKey-upload, #sslCert-upload').click(function(event) {
            event.preventDefault();
            $('#' + $(this).attr('id') + '-field').click();
            return false;
        });

        var commandTracker = this.commandTracker;
        $('#sshKey-upload-field, #sslCert-upload-field').change(function(event) {
            event.preventDefault();
            if (!this || !this.files || this.files.length < 1) {
                return false;
            }
            var fileObject = this.files[0];
            if (!fileObject) {
                return false;
            }
            var filePath = $(this).val();
            var source = $(this);
            var sourceId = this.id;
            if (filePath) {
                var fileName = filePath.split(/(\\|\/)/g).pop();
                $('#sshKey-upload, #sslCert-upload').prop('disabled', true);
                commandTracker.uploadFile(model.get(), fileObject,
                    function(command) {
                        var data = command.response;
                        $('#sshKey-upload, #sslCert-upload').prop('disabled', false);
                        if (data && data.fileName) {
                            var targetValueId = 'data.credentials.' + (sourceId.indexOf('ssl') === 0 ? 'sslCertValue' : 'sshKeyValue');
                            var targetPathId = 'data.credentials.' + (sourceId.indexOf('ssl') === 0 ? 'sslCert' : 'sshKey');
                            model.set(targetValueId, data.fileName);
                            model.set(targetPathId, fileName);
                        }
                    },
                    function() {
                        source.replaceWith(source.clone(true));
                    }
                );
            }
        });

        $('#testConnection-button').click(function(e) {
            $('#testConnection-button').prop('disabled', true);
            $('#testConnection-button').text('Testing...');
            commandTracker.sendCommand('test-connection', model.get(), true);
        });

        this.requestListener = commandTracker.on('completed', scopeId, function(name, command) {
            if (command.name === 'test-connection') {
                $('#testConnection-button').prop('disabled', false);
                $('#testConnection-button').text('Test Connection...');
            }
        });
    };

    CredentialsSettings.prototype.dispose = function(event, el, $) {
        this.commandTracker.off(this.requestListener);
        this.requestListener = null;
    };

    CredentialsSettings.prototype.update = function(event, el, $) {
        var self = this;
        event.changed('data.credentials', function(previous, current) {
            self.updateFields(event.get('data'), el, $);
            self.updateTestButton(event.get('data.url'), current, el, $);
        });
        event.changed('data.url', function(previous, current) {
            self.updateTestButton(current, event.get('data.credentials'), el, $);
        });
    };

    CredentialsSettings.prototype.updateFields = function(data, el, $) {
        var credentials = data.credentials;
        var type = credentials.type;
        var useCache = credentials.useCache;
        var useSshAgent = credentials.useSshAgent;

        $('.credentials-type').val(type);
        $('#useCache').prop('checked', useCache);
        $('#userName').val(credentials.userName);
        $('#password').val(credentials.password);
        $('#passphrase').val(credentials.passphrase);
        $('#sslPassword').val(credentials.sslPassword);
        $('#useSshAgent').prop('checked', credentials.useSshAgent);
        $('#sshKey').val(credentials.sshKey);
        $('#sshKeyValue').val(credentials.sshKeyValue);
        $('#sslCert').val(credentials.sslCert);
        $('#sslCertValue').val(credentials.sslCertValue);
        $('#sshKey-view').text(credentials.sshKey || '<not selected>');
        $('#sslCert-view').text(credentials.sslCert || '<not selected>');

        var cachePath = credentials.cachePath;
        if (!cachePath) {
            cachePath = data.defaultCachePath;
        }
        $('#use-default-credentials-group div.description').html(
            'Configuration directory path: \'' + cachePath + '\' ' +
            '(<a target="_blank" href="http://subgit.com/stash#credentials">learn more</a>).');

        $('select, input[id!="useCache"], button[id!="testConnection-button"]').each(function() {
            $(this).prop('disabled', useCache);
        });
        $('.pathValue').each(function() {
            $(this).attr('disabled', useCache);
        });
        $('#sshKey-group input, #sshKey-group button').each(function() {
            $(this).prop('disabled', useSshAgent || useCache);
        });
        $('#sshKey-group .pathValue').each(function() {
            $(this).attr('disabled', useSshAgent || useCache);
        });

        $('#password').parent().toggle(type === 'SIMPLE');
        $('#sslPassword').parent().toggle(type === 'SSL');
        $('#passphrase').parent().toggle(type === 'SSL' || type === 'SSH');
        $('#sslCert-group').toggle(type === 'SSL');
        $('#sshKey-group').toggle(type === 'SSH');
        $('#use-ssh-agent-group').toggle(type === 'SSH');

        if (type === 'SSL') {
            $("label[for='userName'] > span").removeClass('icon-required');
        } else {
            $("label[for='userName'] > span").removeClass('icon-required').addClass('icon-required');
        }
    };

    CredentialsSettings.prototype.updateTestButton = function(url, credentials, el, $) {
        $('#testConnection-button').prop('disabled', !(Validation.isUrlValid(url) && Validation.isCredentialsValid(credentials)));
    };

    return CredentialsSettings;
});