define('tmatesoft/svn/views/repository-item', [
    '@atlassian/aui',
    'bitbucket/util/navbuilder',
    'tmatesoft/svn/views/status-util',
    'tmatesoft/svn/messages'
], function(
    AJS,
    nav,
    StatusUtil,
    Messages
) {
    'use strict';

    function RepositoryItem(item, pageScope, commandTracker) {
        this.commandTracker = commandTracker;
        this.isShowProject = pageScope.projectId === 0;
        this.scopeId = item.scope.id;
        this.modelPath = 'children.' + this.scopeId;
        this.projectUrl = nav.pluginServlets().path('svn', 'projects', item.data.projectKey).build();
        this.repositoryUrl = nav.pluginServlets().path('svn', 'projects', item.data.projectKey, 'repos', item.data.repositorySlug).build();
    };

    RepositoryItem.prototype.render = function(model, el, $) {
        var data = model.get(this.modelPath).data;

        el.innerHTML = tmatesoft.svn.repositoryItemRow({
            data : data,
            showProject : this.isShowProject,
            projectUrl : this.projectUrl,
            repositoryUrl : this.repositoryUrl
        });
    };

    RepositoryItem.prototype.setup = function(model, el, $) {
        var item = model.get(this.modelPath);
        this.updateButton(item, el, $);
        this.updatePrimaryBadge(item, el, $);
        this.updateSecondaryBadge(item, el, $);
        this.updateMessage(model.get(this.modelPath), $);

        var commandTracker = this.commandTracker;
        var modelPath = this.modelPath;
        var self = this;

        $('button').click(function(e) {
            e.preventDefault();
            var command = $('button').attr('data-command');
            commandTracker.sendCommand(command, model.get(modelPath));
        });

        this.sentListener = commandTracker.on('sent', this.scopeId, function(event, command) {
            $('button').prop('disabled', true);
            if (command.cancelling) {
                $('button').text('Cancelling...');
            } else {
                $('button').attr('data-command', 'cancel');
                $('button').text('Sending...');
            }
            $('button').attr('title', null);
        });
        this.submitListener = commandTracker.on('submitted', this.scopeId, function(event, command) {
            if (!command.cancelling) {
                $('button').text('Cancel');
                $('button').attr('title', null);
            }
            $('button').prop('disabled', false);
        });
        this.completedListener = commandTracker.on('cancelled completed', this.scopeId, function() {
            self.updateButton(model.get(modelPath), el, $);
        });
    };

    RepositoryItem.prototype.update = function(event, el, $) {
        if (event.changed(this.modelPath + '.data.stage') || event.changed(this.modelPath + '.data.importRequested')) {
            var item = event.get(this.modelPath);
            this.updateButton(item, el, $);
            this.updatePrimaryBadge(item, el, $);
            this.updateMessage(event.get(this.modelPath), $);
        }
        if (event.changed(this.modelPath + '.data.hasUnsyncedRefs')
            || event.changed(this.modelPath + '.task.state')) {
            this.updateSecondaryBadge(event.get(this.modelPath), el, $);
        }
        if (event.changed(this.modelPath + '.task') || event.changed(this.modelPath + '.pending')) {
            this.updateMessage(event.get(this.modelPath), $);
        }
    };

    RepositoryItem.prototype.dispose = function() {
        this.commandTracker.off(this.sentListener);
        this.commandTracker.off(this.submitListener);
        this.commandTracker.off(this.completedListener);

        this.sentListener = null;
        this.submitListener = null;
        this.completedListener = null;
    };

    RepositoryItem.prototype.updateButton = function(item, el, $) {
        var stage = item.data.stage;

        var commandRunning = this.commandTracker.isCommandSubmitted(item.scope.id);
        if (!commandRunning) {
            var command = StatusUtil.getCommandForStage(item);
            $('button').text(command.label);
            $('button').attr('data-command', command.name);
            $('button').prop('disabled', command.name === 'unknown');
            $('button').attr('title', command.title);
        }
    };

    RepositoryItem.prototype.updatePrimaryBadge = function(item, el, $) {
        StatusUtil.updatePrimaryBadge(item, $, true, this.repositoryUrl);
    };

    RepositoryItem.prototype.updateSecondaryBadge = function(item, el, $) {
        StatusUtil.updateSecondaryBadge(item, $, true, this.repositoryUrl);
    };

    RepositoryItem.prototype.updateMessage = function(item, $) {
        StatusUtil.updateMessage(item, $);
    };

    return RepositoryItem;
});