define('tmatesoft/svn/views/command-view', [
    'bitbucket/util/navbuilder',
    'tmatesoft/svn/messages',
    'tmatesoft/svn/views/util'
], function(
    nav,
    Messages,
    Util
) {

    function CommandView(commandTracker) {
        this.commandTracker = commandTracker;
    };

    CommandView.prototype.render = function(model, el, $) {
        var self = this;
        this.scopeId = model.get('scope.id');
        this.commandTracker.setCommandConfirmation(function(commandName, item, callback) {
            self.confirmCommand(commandName, item, $, callback);
        });

        var html = '';

        html += '<div class="flag-container">';
        html += '<div class="command-flag"></div>';
        html += '<div class="request-flag"></div>';
        html += '</div>';

        html += '<div class="error-dialog"></div>';
        html += '<div class="confirm-dialog"></div>';

        html += '<div class="command-progress-container">';
        html += tmatesoft.svn.commandProgress();
        html += '</div>';
        html += '<div class="dimmer aui-blanket" tabindex="0" style="position: absolute;" aria-hidden="false"></div>';

        el.innerHTML = html;

        this.toggleProgress(false, {scopeId : this.scopeId}, $);
    };

    CommandView.prototype.toggleProgress = function(visible, command, $, eventName) {
        if (command.scopeId !== this.scopeId) {
            return;
        }
        if (eventName === 'sent') {
            $('#cancel').prop('disabled', true);
            $('#cancel').text('Sending...');
            $('.message').text('Waiting for queued commands to complete...');
        } else if (eventName === 'submitted') {
            $('#cancel').prop('disabled', command.name === 'bootstrap');
            $('#cancel').text('Cancel');
        } else {
            $('#cancel').prop('disabled', true);
            $('#cancel').text('Cancel');
        }
        if (!visible) {
            if (this.progressTimeout) {
                window.clearTimeout(this.progressTimeout);
                this.progressTimeout = null;
            }
            $('.dimmer').toggle(visible);
            $('.command-progress-container').toggle(visible);
        } else {
            if (this.progressTimeout == null) {
                this.progressTimeout = window.setTimeout(function() {
                    $('.dimmer').toggle(visible);
                    $('.command-progress-container').toggle(visible);
                }, 500);
            }
        }
    };

    CommandView.prototype.setup = function(model, el, $) {
        var self = this;
        this.commandListener = this.commandTracker.on('completed cancelled error submitted sent', '',
            function(name, command) {
                name = 'on' + name.substring(0, 1).toUpperCase() + name.substring(1);
                if (self[name]) {
                    self[name](command, $, model);
                }
            });

        $('#cancel').click(function() {
            $(this).prop('disabled', true);
            $(this).text('Cancelling...');
            self.commandTracker.sendCommand('cancel', model.get());
        });
        this.updateDimmerOnPendingTasks(model, $);
    };

    CommandView.prototype.update = function(event, el, $) {
        if (event.changed('task') || event.changed('pending') || event.changed('data.stage')) {
            this.updateDimmerOnPendingTasks(event, $);
        }
        this.commandTracker.update(event);
        if (event.changed('task') || event.changed('pending') || event.changed('data.stage')) {
            var status = Messages.getStatusData(event.get());
            $('.message').text(status.message ? status.message : '');
            $('.progress-title').text(Messages.getProgressTitle(event.get()));
            $('.command-progress-container .bar').css('width', status.percentage + '%');
        }
    };

    CommandView.prototype.updateDimmerOnPendingTasks = function(model, $) {
        var stage = model.get('data.stage');
        if ('INITIAL' === stage || 'CONFIGURED' === stage) {
            var pending = model.get('pending') || [];
            var task = model.get('task');
            if (!(task && task.state === 'RUNNING')) {
                task = null;
            }
            if (task !== null || (pending.length > 0)) {
                $('.dimmer').toggle(true);
            } else {
                $('.dimmer').toggle(false);
            }
        } else {
            $('.dimmer').toggle(false);
        }
    };

    CommandView.prototype.dispose = function(model, el, $) {
        this.commandTracker.off(this.commandListener);
        this.commandListener = null;
    };

    CommandView.prototype.onCompleted = function(command, $, model) {
        this.toggleProgress(false, command, $, 'completed');
        if (command.name === 'test-connection') {
            var response = command.response;
            if (!response.readAccess) {
                var errorMessage = response.error || 'Unknown Error';
                this.showRequestFlag('error', 'Connection Test Failed', errorMessage, $);
            } else if (!response.writeAccess) {
                var errorMessage = response.error || 'Unknown Error';
                this.showRequestFlag('warning', 'No Write Access', errorMessage, $);
            } else {
                this.showRequestFlag('success', 'Connection Test Successful', '<p>Read Access: OK<br/>Write Access: OK</p>', $);
            }
        } else if (command.name === 'build-support-zip') {
            var response = command.response;
            if (response && response.error) {
                this.showRequestFlag('error', 'Create Support Zip Failed', response.error, $);
            } else {
                var zipUrl = window.location.pathname + '/support/' + response.name;
                var zipSize = Messages.formatFileSize(response.size);
                this.showRequestFlag('success',
                  'Support Zip Created',
                  '<a href="' + zipUrl + '" download="' + response.name + '">' + response.name + '</a> (' + zipSize + ')',
                  $);
            }
        } else if (command.name === 'remove-support-zips') {
            var response = command.response;
            if (response && response.error) {
                this.showRequestFlag('error', 'Remove Support Zips Failed', response.error, $);
            } else if (response && response.filesRemoved > 0) {
                var plural = ((response.filesRemoved > 1) ? 's' : '');
                this.showRequestFlag('success', 'Support Zip' + plural + ' Removed',
                response.filesRemoved + ' file' + plural + ' removed', $);
            }
        } else if (command.name === 'upload-file') {
            var response = command.response;
            if (response && response.error) {
                this.showRequestFlag('error', 'Upload Failed', response.error, $);
            }
        } else if (command.error) {
            var specialData = this.getSpecialErrorData(command.error);
            if (specialData !== null && specialData.code === 'EVALUATION_REVISION_COUNT_EXCEEDED') {
                this.displayEvaluationRevisionCountError(specialData, $);
            } else {
                var item = model.get('scopeId' === command.scopeId) ? model.get() : model.get('children.' + command.scopeId);
                var flagData = Messages.getCommandFlagData(command, item);
                if (flagData) {
                    this.showCommandFlag('error', flagData.title, flagData.content, $);
                }
            }
        } else {
            if (command.name === 'apply' || command.name === 'configure') {
                model.revert('data.credentials');
            }
            var item = model.get('scopeId' === command.scopeId) ? model.get() : model.get('children.' + command.scopeId);
            var flagData = Messages.getCommandFlagData(command, item);
            if (flagData) {
                this.showCommandFlag('success', flagData.title, flagData.content, $);
            }
        }
    };

    CommandView.prototype.onSent = function(command, $) {
        this.toggleProgress(true, command, $, 'sent');
    };

    CommandView.prototype.onSubmitted = function(command, $) {
        this.toggleProgress(true, command, $, 'submitted');
    };

    CommandView.prototype.onCancelled = function(command, $, model) {
        this.toggleProgress(false, command, $, 'cancelled');

        if (!command.unknown) {
            var item = model.get('scopeId' === command.scopeId) ? model.get() : model.get('children.' + command.scopeId);
            var flagData = Messages.getCommandFlagData(command, item);
            this.showCommandFlag('generic', flagData.title, flagData.content, $);
        }
    };

    CommandView.prototype.confirmCommand = function(commandName, item, $, callback) {
        var confirmationData = Messages.getConfirmationData(commandName, item);
        if (confirmationData === null) {
            callback();
        } else {
            var dialog = Util.createDialog($('.confirm-dialog')[0], $, 'confirmation',
                confirmationData.title,
                confirmationData.content,
                [
                {
                    id : 'confirm-button',
                    type : 'primary',
                    label : confirmationData.label
                },
                {
                    id : 'cancel-button',
                    label : 'Cancel'
                },
                ]
            );
            $('#confirm-button').click(function() {
                dialog.hide();
                callback();
            });
            $('#cancel-button').click(function() {
                dialog.hide();
            });
            dialog.show();
        }
    };

    CommandView.prototype.onError = function(command, $) {
        this.toggleProgress(false, command, $);
        this.commandTracker.stopPollers();
        var dialog = Util.createDialog($('.error-dialog')[0], $, 'warning',
            'Unexpected Error',
            '<p>Error has occurred that was not expected at the moment:</p>' +
            '<pre>' + (command.error || 'Unknown Error') + '</pre>',
            [{
                id : 'reload-button',
                label : 'Reload'
            }]);
        $('#reload-button').click(function() {
            location.reload(true);
        });

        dialog.show();
    };

    CommandView.prototype.showCommandFlag = function(type, title, message, $) {
        this.showFlag('command-flag', type, title, message, $);
    };

    CommandView.prototype.showRequestFlag = function(type, title, message, $) {
        this.showFlag('request-flag', type, title, message, $);
    };

    CommandView.prototype.clearFlagTimeout = function(flagName) {
        if (this[flagName + '-closer']) {
            window.clearTimeout(this[flagName + '-closer']);
            delete this[flagName + '-closer'];
        }
    };

    CommandView.prototype.closeFlag = function(flagName, $, timeout) {
        this.clearFlagTimeout(flagName);
        if (timeout) {
            var self = this;
            this[flagName + '-closer'] = window.setTimeout(function() {
                self.closeFlag(flagName, $);
            }, 10*1000);
            return;
        }
        $('.' + flagName+ ' div').hide(400, function() {
            $('.' + flagName).empty();
        });
    };

    CommandView.prototype.showFlag = function(flagName, type, title, message, $) {
        this.clearFlagTimeout(flagName);
        $('.' + flagName).empty();
        var html = tmatesoft.svn.flag({ type : type, title : title, message : message, closeable : true});
        $('.' + flagName).append(html);
        if (type == 'success') {
            this.closeFlag(flagName, $, 7 * 1000);
        }
        var self = this;
        $('.' + flagName + ' .icon-close').click(function() {
            self.closeFlag(flagName, $);
        });
    };

    CommandView.prototype.getSpecialErrorData = function(error) {
        if (error === null) {
            return null;
        }
        var match = error.match(/^(.+)\?(.*)$/);
        if (match === null || match.length < 3) {
            return null;
        }
        var params = match[2].split('&');
        var code = match[1];
        var result = {
            code : code,
            data : {}
        };

        for(var i = 0; params !== null && i < params.length; i++) {
            var kv = params[i].split('=');
            if (kv !== null && kv.length >= 2) {
                result.data[kv[0]] = kv[1];
            }
        }
        return result;
    };

    CommandView.prototype.displayEvaluationRevisionCountError = function(data, $) {
        var dialog = Util.createDialog($('.error-dialog')[0], $, 'warning',
            'Evaluation Limit Exceeded',
            '<p>You are running SVN Mirror Add-on in <b>Evaluation Mode</b>.</p>' +
            '<p>' +
            ' In Evaluation Mode SVN mirror and import operations are limited to SVN projects' +
            ' with up to <b>10000 revisions</b>.' +
            ' Your SVN project history contains <b>' + data.data['revisionCount'] + ' revisions</b>.</p>' +
            '<p>To set up a Git/SVN mirror or import' +
            ' SVN project with more than 10000 revisions, get a production license for the SVN Mirror Add-on.' +
            '</p>',
            [
            {
                id : 'license-button',
                label : 'Get License',
                type : 'primary'
            },
            {
                id : 'cancel-button',
                label : 'Cancel'
            }]);

        $('#cancel-button').click(function() {
            dialog.hide();
            dialog.dispose();
        });

        $('#license-button').click(function() {
            var targetUrl = nav.pluginServlets().path('upm').withFragment('manage/org.tmatesoft.subgit.stash-svn-importer').build();
            window.location.href = targetUrl;
        });

        dialog.show();
    };

    return CommandView;
});