define('tmatesoft/svn/views/buttons-view', [
    'tmatesoft/engine-util',
    'tmatesoft/svn/validation',
    'tmatesoft/svn/views/util'
], function(
    Util,
    Validation,
    ViewUtil
) {
    'use strict';

    function ButtonsView(commandTracker) {
        this.commandTracker = commandTracker;
    };

    var encryptedFields = ['data.credentials.password',
                           'data.credentials.sslPassword',
                           'data.credentials.passphrase'];

    ButtonsView.prototype.render = function(model, el, $) {
        var html = '';
        html += '<button class="aui-button uninstall" data-command="reset" title="Reset configuration and start from scratch">Back</button>';

        html += '<div class="aui-buttons">';
        html += '<button class="aui-button aui-button-split-main aui-button-primary apply"></button>';
        html += '<button class="aui-button aui-dropdown2-trigger aui-button-primary aui-button-split-more apply-split" aria-controls="split-container-dropdown">Split more</button>';
        html += '</div>';
        html += '<aui-dropdown-menu id="split-container-dropdown">';
        html += '<aui-item-link class="apply-rebuild">Apply Changes and Rebuild Mirror</aui-item-link>';
        html += '</aui-dropdown-menu>';

        html += '<button class="aui-button aui-button-link revert"></button>';
        html += '<span style="margin-left: 1em;"></span>';
        html += '<button class="aui-button aui-button import" data-command="start-import" title="Perform One-Time Import">Import</button>';
        html += '<button class="aui-button aui-button-primary start" data-command="start" title="Start SVN Mirror">Mirror</button>';
        html += '<div class="confirm-refetch-dialog"></div>';
        el.innerHTML = html;
    };

    ButtonsView.prototype.setup = function(model, el, $) {
        var self = this;
        $('.revert').click(function() {
            model.revert();
        });
        $('.apply,.uninstall,.start,.import').click(function() {
            var command = $(this).attr('data-command');
            var commandData = model.get();
            var stage = model.get('data.stage');
            var commandData = Util.clone(commandData);

            if ((command === 'start' || command === 'apply') && (stage === 'INSTALLED' || stage === 'PAUSED')) {
                var oldConfig = model.getRemote('data.config');
                var newConfig = model.get('data.config');
                var addedMappings = Validation.getAddedMappings(oldConfig, newConfig);
                if (addedMappings.length > 0) {
                    self.showRefetchConfirmationDialog(addedMappings, $, function() {
                        commandData.data.forceRefetch = true;
                        commandData.data.forceRebuild = false;
                        self.commandTracker.sendCommand(command, commandData);
                    });
                    return;
                }
            }
            for(var i = 0; i < encryptedFields.length; i++) {
                var newValue = model.get(encryptedFields[i]);
                var oldValue = model.getRemote(encryptedFields[i]);
                if (newValue !== oldValue) {
                    model.set(encryptedFields[i], oldValue);
                }
            }
            commandData.data.forceRefetch = false;
            commandData.data.forceRebuild = false;
            self.commandTracker.sendCommand(command, commandData);
            // TODO: set encrypted fields back to user values on that particular command failure
        });

        $('.apply-split').click(function() {
            if (!$(this).hasClass('aui-dropdown2-active')) {
                $.$('.apply-rebuild a').one('click.svn', function() {
                    var commandData = Util.clone(model.get());
                    for(var i = 0; i < encryptedFields.length; i++) {
                        var newValue = model.get(encryptedFields[i]);
                        var oldValue = model.getRemote(encryptedFields[i]);
                        if (newValue !== oldValue) {
                            model.set(encryptedFields[i], oldValue);
                        }
                    }
                    commandData.data.forceRefetch = false;
                    commandData.data.forceRebuild = true;

                    self.commandTracker.sendCommand('apply', commandData);
                    return true;
                });
            } else {
                $.$('.apply-rebuild a').off('click.svn');
            }
        });

        this.commandListener = this.commandTracker.on('sent completed cancelled', model.get('scope.id'), function() {
            self.updateButtonsState(model, $);
        });

        this.updateButtonsText(model.get('data.stage'), $);
        this.updateButtonsState(model, $);
    };

    ButtonsView.prototype.update = function(event, el, $) {
        var self = this;
        event.changed('data.stage', function(previous, current) {
            self.updateButtonsText(current, $);
        });
        this.updateButtonsState(event.model, $);
    };

    ButtonsView.prototype.dispose = function(event, el, $) {
        this.commandTracker.off(this.commandListener);
        this.commandListener = null;
    };

    ButtonsView.prototype.updateButtonsState = function(model, $) {
        var stage = model.get('data.stage');
        if (this.commandTracker.isCommandSubmitted(model.get('scope.id'))) {
            $('.apply, .apply-split, .revert, .uninstall, .start, .import').prop('disabled', true);
        } else {
            var hasChanges = model.hasChanges();
            var modelValid = Validation.isModelValid(model.get());
            $('.apply').prop('disabled', (!hasChanges && stage !== 'INITIAL') || !modelValid);
            $('.apply-split').prop('disabled', (!hasChanges && stage !== 'INITIAL') || !modelValid);
            $('.revert').prop('disabled', !hasChanges);
            $('.uninstall').prop('disabled', false);

            $('.start').prop('disabled', !modelValid);
            $('.import').prop('disabled', !modelValid);
        }

        $('.apply').toggle(stage === 'INITIAL' || stage === 'PAUSED' || stage === 'INSTALLED' || stage === 'GLOBAL');
        $('.apply-split').toggle(stage === 'PAUSED' || stage === 'INSTALLED');
        $('.uninstall').toggle(stage === 'CONFIGURED');
        $('.start').toggle(stage === 'CONFIGURED');
        $('.import').toggle(stage === 'CONFIGURED');
    };

    ButtonsView.prototype.updateButtonsText = function(stage, $) {
        if (stage === 'INITIAL') {
            $('.apply').text('Continue').attr('data-command', 'configure');
            $('.revert').text('Reset to Defaults');
        } else {
            $('.apply').text('Apply Changes').attr('data-command', 'apply');
            $('.revert').text('Revert Changes');
        }
    };

    ButtonsView.prototype.showRefetchConfirmationDialog = function(addedMappings, $, callback) {
        var listOfLocations = '<code><br/>';
        for (var i = 0; i < addedMappings.length; i++) {
            listOfLocations += addedMappings[i] + '<br/>';
        }
        var singleMapping = addedMappings.length == 1;
        listOfLocations += '<br/></code>';
        var dialog = ViewUtil.createDialog($('.confirm-refetch-dialog')[0], $, 'confirmation',
            'Import existing SVN revisions',
            'Branches mapping has been modified and the following SVN/Git branches ' +
            'mapping' + (singleMapping ? ' is' : 's are') + ' about to become effective:<br/>' +
            listOfLocations +
            'SVN branches or tags that match new mapping' + (singleMapping ? '' : 's') +
            ', will be imported from SVN project to this Git repository. <b>Import ' +
            'may take significant time</b>, depending on the total size of SVN repository',
            [
            {
                id : 'refetch-button',
                type : 'primary',
                label : 'Apply Changes and Import'
            },
            {
                id : 'cancel-button',
                label : 'Cancel'
            },
            ]
        );
        $('#refetch-button').click(function() {
            dialog.hide();
            callback();
        });
        $('#cancel-button').click(function() {
            dialog.hide();
        });
        dialog.show();
    };

    return ButtonsView;
});