define('tmatesoft/svn/validation', [
    'exports'
], function(
    exports
) {

    'use strict';

    var isModelValid = function(model) {
        var stage = model.data.stage;
        if (stage === 'INITIAL') {
            return isUrlValid(model.data.url)
                && isDomainValid(model.data.defaultDomain)
                && isCredentialsValid(model.data.credentials)
                && isLayoutValid(model.data.layoutType, model.data.trunkPath);
        } else if (stage === 'PAUSED' || stage === 'INSTALLED' || stage === 'CONFIGURED') {
            return isUrlValid(model.data.url)
                && isDomainValid(model.data.defaultDomain)
                && isFetchIntervalValid(model.data.fetchInterval)
                && isMinimalRevisionValid(model.data.minimalRevision)
                && isCredentialsValid(model.data.credentials)
                && isConfigValid(model.data.config);
        } else if (stage === 'GLOBAL') {
            return isSchedulerOptionsValid(model.data.schedulerOptions)
                && isPollOptionsValid(model.data.longPollOptions)
                && isAuthorisationOptionsValid(model.data.authorisationOptions);
        }
        return true;
    };

    var isUrlValid = function(url) {
        if (!isNonEmptyString(url)) {
            return false;
        }
        var regex = /^(https?|svn(\+[^\/]+)?|file):\/\/(([^\/:]+(:[0-9]+)?)|\/)(.*)?$/;
        return url.match(regex);
    };

    var isSchedulerOptionsValid = function(options) {
        var runnersCount = parseInt(options.maxRunnersCount, 10);
        return runnersCount === -1 || (!isNaN(runnersCount) && runnersCount > 0);
    };

    var isFetchIntervalValid = function(interval) {
        var interval = parseInt(interval, 10);
        return !isNaN(interval) && interval >= 0;
    };

    var isMinimalRevisionValid = function(revision) {
        var revision = parseInt(revision, 10);
        return !isNaN(revision) && revision >= 0;
    };

    var isRebuildRevisionValid = function(revision, minimalRevision) {
        var revision = parseInt(revision, 10);
        return !isNaN(revision) && revision >= minimalRevision;
    };

    var isPollOptionsValid = function(options) {
        if (!options.enabled) {
            return true;
        }
        var interval = parseInt(options.interval, 10);
        return !isNaN(interval) && interval >= 10 && interval <= 600;
    };

    var isAuthorisationOptionsValid = function(options) {
        var mirrorGroup = options.mirrorGroupName;
        var importGroup = options.importGroupName;
        return (mirrorGroup !== null && mirrorGroup !== '')
            && (importGroup !== null && importGroup !== '')
    };

    var isConfigValid = function(config) {
        return isNonEmptyString(config);
    };

    var isTrunkPathValid = function(path) {
        return isNonEmptyString(path);
    };

    var isDomainValid = function(domain) {
        return isNonEmptyString(domain);
    };

    var isCredentialsValid = function(credentials) {
        if (!credentials) {
            return false;
        }
        if (credentials.useCache) {
            return true;
        } else if (credentials.type === 'SIMPLE') {
            return isNonEmptyString(credentials.userName);
        } else if (credentials.type === 'SSH') {
            return isNonEmptyString(credentials.userName)
                && (credentials.useSshAgent || isNonEmptyString(credentials.sshKey));
        } else if (credentials.type === 'SSL') {
            return isNonEmptyString(credentials.sslCert);
        }
        return false;
    };

    var isLayoutValid = function(layoutType, trunkPath) {
        if (layoutType === 'AUTO') {
            return isTrunkPathValid(trunkPath);
        }
        return true;
    }

    var isNonEmptyString = function(value) {
        return value && value !== null && value.trim() !== '';
    };

    var getAddedMappings = function(oldConfigValue, newConfigValue) {
        var oldMappings = parseMapping(oldConfigValue);
        var newMappings = parseMapping(newConfigValue);
        var addedMappings = [];
        for (var i = 0; i < newMappings.length; i++) {
            if (oldMappings.indexOf(newMappings[i]) < 0) {
                addedMappings.push(newMappings[i]);
            }
        }
        return addedMappings;
    };

    var parseMapping = function(data) {
        var regex = {
            section: /^\s*\[\s*(.*)\s*\]\s*$/,
            param: /^\s*(branches|tags|trunk|shelves)\s*=\s*(.*)\s*$/
        };

        var value = {};
        var lines = data.split(/\r\n|\r|\n/);

        var mappings = [];
        var mappingSection = false;

        lines.forEach(function(line) {
            if (regex.param.test(line) && mappingSection) {
                var match = line.match(regex.param);
                mappings.push(match[2]);
            } else if (regex.section.test(line)) {
                var match = line.match(regex.section);
                var section = match[1];
                if (section === 'svn') {
                    mappingSection = true;
                } else {
                    mappingSection = false;
                }
            }
        });
        return mappings;
    };


    exports.getAddedMappings = getAddedMappings;
    exports.isModelValid = isModelValid;
    exports.isCredentialsValid = isCredentialsValid;
    exports.isUrlValid = isUrlValid;
    exports.isRebuildRevisionValid = isRebuildRevisionValid;
    exports.isMinimalRevisionValid = isMinimalRevisionValid;

});
