define('tmatesoft/svn/command-tracker', [
    '@atlassian/aui',
    'bitbucket/util/navbuilder',
    'tmatesoft/svnmirror/util/poll',
], function(
    AJS,
    nav,
    poller
) {

    function CommandTracker(codec) {
        this.codec = codec;
        this.commands = {};
        this.listeners = {};
        this.pollers = [];
    };

    CommandTracker.prototype.stopPollers = function() {
        for(var i = 0; i < this.pollers.length; i++) {
            this.pollers[i].stop();
        }
        this.pollers = [];
    };

    CommandTracker.prototype.updatePollOptions = function(options) {
        for(var i = 0; i < this.pollers.length; i++) {
            var oldPoller = this.pollers[i];
            this.pollers[i] = this.clonePoll(oldPoller, options);
            oldPoller.stop();
            delete oldPoller;
            this.pollers[i].poll();
        }
    };

    CommandTracker.prototype.setCommandConfirmation = function(confirmation) {
        this.confirmation = confirmation;
    };

    CommandTracker.prototype.on = function(events, scopeId, listener) {
        var names = events.split(' ');
        scopeId = scopeId || '';
        for(var i = 0; i < names.length; i++) {
            var name = names[i];
            this.listeners[name] = this.listeners[name] || {};
            this.listeners[name][scopeId] = this.listeners[name][scopeId] || [];
            this.listeners[name][scopeId].push(listener);
        }
        return listener;
    };

    CommandTracker.prototype.off = function(listener) {
        if (!listener) {
            return;
        }
        for(var name in this.listeners) {
            if (!this.listeners.hasOwnProperty(name)) {
                continue;
            }

            for(var scope in this.listeners[name]) {
                if (!this.listeners[name].hasOwnProperty(scope)) {
                    continue;
                }
                var registered = this.listeners[name][scope];
                while(registered && registered.indexOf(listener) >= 0) {
                    registered.splice(registered.indexOf(listener), 1);
                }
            }
        }
    };

    CommandTracker.prototype.triggerEvent = function(name, command) {
        var scopeId = command.scopeId;
        var listenersForEvent = this.listeners[name];
        if (listenersForEvent) {
            for(var listenerScopeId in listenersForEvent) {
                if (!listenersForEvent.hasOwnProperty(listenerScopeId)) {
                    continue;
                }
                if (listenerScopeId === '' || listenerScopeId === scopeId) {
                    var ls = listenersForEvent[listenerScopeId];
                    for(var i = 0; i < ls.length; i++) {
                        ls[i](name, command);
                    }
                }
            }
        }
    };

    CommandTracker.prototype.uploadFile = function(item, file, okCb, completeCb) {
        var scopeId = item.scope.id;
        var self = this;
        var url = this.buildCommandUrl('upload-file', item);

        var formData = new FormData();
        formData.append('file', file);

        var command = {
            name : 'upload-file',
            scopeId : scopeId
        };
        AJS.$.ajax({
            url : url,
            data : formData,
            processData : false,
            contentType : false,
            dataType : 'json',
            type : 'POST',
            success : function(data) {
                command.response = self.codec ? self.codec.decode(data) : data;
                okCb(command);
                self.triggerEvent("completed", command);
            },
            error : function(xhr, status, err) {
                command.error = 'Connection Failed';
                self.triggerEvent('error', command);
            },
            complete : completeCb
        });
    };

    CommandTracker.prototype.clonePoll = function(p, options) {
        var clone = poller.poll('?command=poll', '?command=get', options);
        clone.receiver = p.receiver;
        clone.errorReceiver = p.errorReceiver;
        clone.logReceiver = p.logReceiver;
        return clone;
    }

    CommandTracker.prototype.createPoll = function(scopeId, pollOptions, receiver, logReceiver) {
        var self = this;
        var p = poller.poll('?command=poll', '?command=get', pollOptions);
        p.data(function(delta, isComplete) {
            delta = self.codec ? self.codec.decode(delta) : delta;
            if (!delta || delta.error) {
                var data = {
                    name : 'poll',
                    scopeId : scopeId,
                    error : delta ? delta.error : 'Unknown Error'
                };
                self.triggerEvent('error', data);
            } else {
                receiver(delta, isComplete);
            }
        });
        p.log(function(message) {
            if (logReceiver) {
                var data = {
                    scopeId : scopeId,
                    data : {
                        stage : 'LOG',
                        message : message
                    }
                };
                logReceiver(data);
            }
        });
        p.error(function(message) {
            var data = {
                name : 'poll',
                scopeId : scopeId,
                error : message || 'Connection Lost'
            };
            self.triggerEvent('error', data);
        });
        this.pollers.push(p);
        return p;
    };

    CommandTracker.prototype.sendCommand = function(commandName, item, isRequest, isConfirmed) {
        if (commandName !== 'cancel' && !isRequest && !isConfirmed && this.confirmation) {
            var callback = this.sendCommand.bind(this, commandName, item, false, true);
            this.confirmation(commandName, item, callback);
            return;
        }
        var scopeId = item.scope.id;
        var url = this.buildCommandUrl(commandName, item);
        var data = this.codec ? this.codec.encode(item) : item;
        var command;
        if (commandName === 'cancel') {
            var runningCommand = this.commands[scopeId];
            if (!runningCommand) {
                this.triggerEvent('error', {
                    scopeId : scopeId,
                    error : 'Cancel was not expected at this moment'
                });
                return;
            } else {
                data.task.id = runningCommand.taskId;
                runningCommand.cancelling = true;
                command = runningCommand;
            }
        } else {
            command = this.buildCommandObject(commandName, item);
        }


        var self = this;
        if (!isRequest && commandName !== 'cancel') {
            self.triggerEvent('sent', command);
        }
        AJS.$.ajax({
            url: url,
            type: 'POST',
            data: JSON.stringify(data),
            contentType: 'application/json; charset=utf-8',
            dataType: 'json',
            success : function(data) {
                data = self.codec ? self.codec.decode(data) : data;
                if (command.cancelling) {
                     if (data.taskId !== -1) {
                        if (!isRequest) {
                            delete self.commands[command.scopeId];
                        }
                        command.cancelled = true;
                        self.triggerEvent('cancelled', command);
                     }
                     return;
                }
                command.response = data || {};
                if (!data || data.error) {
                    if (!isRequest) {
                        delete self.commands[command.scopeId];
                    }
                    command.error = data ? data.error : 'Unknown Error';
                    self.triggerEvent('completed', command);
                } else if (!isRequest) {
                    self.commands[scopeId] = command;
                    command.taskId = data.taskId;
                    self.triggerEvent('submitted', command);
                } else {
                    self.triggerEvent('completed', command);
                }
            },
            error: function(xhr, status, error) {
                if (!isRequest) {
                    delete self.commands[command.scopeId];
                }
                command.error = 'Connection Failure';
                self.triggerEvent('error', command);
            }
        });
    };

    CommandTracker.prototype.update = function(event) {
        for(var scopeId in this.commands) {
            if (!this.commands.hasOwnProperty(scopeId)) {
                continue;
            }
            this.updateCommand(event, this.commands[scopeId], scopeId);
        }
    };

    CommandTracker.prototype.isCommandSubmitted = function(scopeId) {
        return this.commands.hasOwnProperty(scopeId);
    };

    CommandTracker.prototype.updateCommand = function(event, command, scopeId) {
        var item;
        if (event.get('scope.id') === scopeId && (event.changed('task') || event.changed('pending'))) {
            item = event.get();
        } else if (event.changed('children.' + scopeId + '.task') || event.changed('children.' + scopeId + '.pending')) {
            item = event.get('children.' + scopeId);
        }
        if (item) {
            if (isCommandCompleted(command, item.task)) {
                delete this.commands[scopeId];
                if (item.task.state === 'FAILED') {
                    command.error = item.task.message || 'Unknown Error';
                }
                this.triggerEvent("completed", command);
            } else if (isCommandCompletedUnknown(command, item.task, item.pending)) {
                delete this.commands[scopeId];
                command.unknown = true;
                this.triggerEvent("cancelled", command);
            } else if (isCommandCancelled(command, item.task, item.pending)) {
                delete this.commands[scopeId];
                command.cancelled = true;
                this.triggerEvent("cancelled", command);
            }
        }
    };

    var isCommandCompleted = function(command, task, pending) {
        return command.taskId === task.id &&
            (task.completed >= 0 && (
                task.state === 'REJECTED'
             || task.state === 'SHUTDOWN'
             || task.state === 'SUCCESS'
             || task.state === 'FAILED'
             || task.state === 'TIMEOUT'
             ));
    };

    var isCommandCompletedUnknown = function(command, task, pending) {
        if (command.taskId < task.id) {
            return true;
        }
        return false;
    };

    var isCommandCancelled = function(command, task, pending) {
        if (command.taskId === task.id) {
            return task.state === 'CANCELLED';
        }
        return false;
    };

    CommandTracker.prototype.buildCommandUrl = function(command, item) {
        var scope = item.scope;
        var builder;
        if (scope.projectId === 0) {
            builder = nav.pluginServlets().path('svn');
        } else if (scope.repositoryId === 0) {
            builder = nav.pluginServlets().path('svn', 'projects', item.data.projectKey);
        } else {
            builder = nav.pluginServlets().path('svn', 'projects', item.data.projectKey, 'repos', item.data.repositorySlug);
        }
        return builder.withParams({command : command}).build();
    };

    CommandTracker.prototype.buildCommandObject = function(command, item) {
        var scope = item.scope;
        var object = {
            name : command,
            scopeId : scope.id
        };
        if (scope.projectId !== 0) {
            object.projectName = item.data.projectName;
        }
        if (scope.repositoryId !== 0) {
            object.repositoryName = item.data.repositoryName;
        }
        return object;
    };

    return CommandTracker;
});