define('tmatesoft/engine', [
    'tmatesoft/engine-util',
    'exports'
], function(
    Util,
    exports) {

    'use strict';

    var DELTA_DELETE_MARKER = Number.POSITIVE_INFINITY;

    function State(model) {
        this.remote = model || {};
        this.local = Util.clone(this.remote);
    };

    State.prototype.revert = function() {
        this.previous = Util.clone(this.local);
        this.local = Util.clone(this.remote);
    };

    State.prototype.getRemote = function(path) {
        return Util.getValueAtPath(this.remote, path);
    }

    State.prototype.getLocal = function(path) {
        return Util.getValueAtPath(this.local, path);
    }

    State.prototype.setLocal = function(path, value) {
        Util.setValueAtPath(this.local, path, value);
    }

    State.prototype.applyDelta = function(delta, isComplete) {
        var trees = {
            remote : this.remote,
            local : this.local,
            delta : delta
        };
        var changedPaths = new Util.FlatObject();
        Util.treesTraverse(trees, function(key, nodes, values, path) {
            if (Util.isObject(values.delta) && Util.isObject(values.local) && Util.isObject(values.remote)) {
                return true;
            } else if (values.delta === undefined && !isComplete) {
                return false;
            }
            if (!isComplete || !Util.valuesEqual(values.remote, values.delta)) {
                var isDelete = (isComplete && values.delta === undefined) || values.delta === DELTA_DELETE_MARKER;
                if (Util.valuesEqual(values.local, values.remote)) {
                    changedPaths.set(path, values.local);
                    isDelete ? delete nodes.local[key] : nodes.local[key] = Util.clone(values.delta);
                } else if (Util.valuesEqual(values.local, values.delta)) {
                    changedPaths.set(path, values.local);
                }
                isDelete ? delete nodes.remote[key] : nodes.remote[key] = values.delta;
            }
            return false;
        });
        return changedPaths;
    };

    function Controller() {
        this.state = new State();
        this.views = [];
        this.changedPaths = new Util.FlatObject();
    };

    Controller.prototype.update = function(delta, isComplete) {
        var updatedPaths = this.state.applyDelta(delta, isComplete);
        if (!updatedPaths.isEmpty()) {
            var updated = updatedPaths.keys();
            for(var i = 0; i < updated.length; i++) {
                this._markUnchanged(updated[i]);
            }
            this.fireEvent(new Event(this, undefined, updatedPaths, true));
        }
    };

    Controller.prototype.revert = function(path) {
        if (path) {
            this.set(path, Util.clone(this.state.getRemote(path)));
        } else {
            var previousState = this.changedPaths;
            this.state.revert();
            this.changedPaths = new Util.FlatObject();

            var event = new Event(this, undefined, previousState);
            this.fireEvent(event);
        }
    };

    Controller.prototype.revertValue = function(path) {
    }

    Controller.prototype.fireEvent = function(event) {
        for(var i = 0; i < this.views.length; i++) {
            this.views[i].update(event);
        }
    };

    Controller.prototype.addView = function(delegate, domElement) {
        if (!delegate || !domElement) {
            return;
        }
        var view = this.createView(delegate, domElement);
        this.views.push(view);
        view.render(this);
        return delegate;
    };

    Controller.prototype.removeView = function(delegate) {
        for(var i = 0; i < this.views.length; i++) {
            if (this.views[i].delegate === delegate) {
                this.views.splice(index, 1);
                this.views[i].dispose(this);
            }
        }
    };

    Controller.prototype.createView = function(delegate, domElement) {
        var view = new View(this, delegate, domElement);
        delegate.el = domElement;
        return view;
    };

    Controller.prototype.get = function(path) {
        return this.state.getLocal(path);
    };

    Controller.prototype.getRemote = function(path) {
        return this.state.getRemote(path);
    };

    Controller.prototype.set = function(path, value, origin) {
        var currentValue = this.state.getLocal(path);
        if (Util.valuesEqual(currentValue, value)) {
            return;
        }
        this.state.setLocal(path, value);

        var remoteValue = this.state.getRemote(path);
        if (Util.valuesEqual(remoteValue, value)) {
            this._markUnchanged(path);
        } else {
            this._markChanged(path, value);
        }

        var previousState = new Util.FlatObject();
        previousState.set(path, currentValue);
        this.fireEvent(new Event(this, origin, previousState));
    };

    Controller.prototype.hasChanges = function() {
        return !this.changedPaths.isEmpty();
    };

    Controller.prototype._markUnchanged = function(path) {
        this.changedPaths.remove(path);
    };

    Controller.prototype._markChanged = function(path, value) {
        this.changedPaths.set(path);
    };

    function Event(model, origin, changedPaths, isRemote) {
        this.model = model;
        this.origin = origin;
        this.changedPaths = changedPaths;
        this.isRemote = isRemote;
    };

    Event.prototype.get = function(path) {
        return this.model.get(path);
    };

    Event.prototype.changed = function(path, cb) {
        var dotNotation = path.charAt(path.length - 1) === '.';
        if (dotNotation) {
            path = path.substring(0, path.length - 1);
        }
        if (!this.changedPaths || !this.changedPaths.has(path)) {
            return false;
        }
        var previousValue = this.changedPaths.get(path);
        var currentValue = this.model.get(path);
        if (Util.isFunction(cb)) {
            if (dotNotation) {
                var keys;
                if (previousValue === undefined || previousValue === null) {
                    keys = Util.getObjectKeySet(currentValue);
                } else {
                    keys = Util.getObjectKeySet(previousValue);
                }
                var result = false;
                for(var objectKey in keys) {
                    if (!keys.hasOwnProperty(objectKey)) {
                        continue;
                    }
                    result |= cb(previousValue ? previousValue[objectKey] : undefined,
                        currentValue ? currentValue[objectKey]: undefined,
                        objectKey);
                }
                return result;
            } else {
                return cb(previousValue, currentValue);
            }
        }
        return true;
    };

    function View(controller, delegate, domElement) {
        this.el = domElement;
        this.delegate = delegate || {};
        this.controller = controller;
        this.childViews = [];
        if (Util.isFunction(this.delegate.children)) {
            this.children = this.delegate.children() || {};
        } else {
            this.children = this.delegate.children || {};
        }
    };

    View.prototype.collectChildElements = function() {
        var placeholders = {};
        for(var childId in this.children) {
            if (!this.children.hasOwnProperty(childId)) {
                continue;
            }
            var domElements = this.el.querySelectorAll(childId);
            if (domElements) {
                var newElements = [];
                for(var i = 0; i < domElements.length; i++) {
                    var domElement = domElements.item(i);
                    if (this.containsElement(domElement) && !this.isChildViewElement(domElement)) {
                        newElements.push(domElement);
                    }
                }
                placeholders[childId] = newElements;
            }
        }
        return placeholders;
    };

    View.prototype.createChildViews = function(placeholders, model) {
        for(var childId in placeholders) {
            if (!placeholders.hasOwnProperty(childId)) {
                continue;
            }
            var childFactory = this.children[childId];
            if (Util.isFunction(childFactory)) {
                var domElements = placeholders[childId];
                for(var i = 0; i < domElements.length; i++) {
                    var childDelegate = childFactory(this.controller, domElements[i]);
                    var childView = this.controller.createView(childDelegate, domElements[i]);
                    this.childViews.push(childView);
                    childView.render(model);
                }
            }
        }
        return placeholders;
    };

    View.prototype.render = function(model) {
        if (Util.isFunction(this.delegate.render)) {
            this.delegate.render(model, this.el);
            this.createChildViews(this.collectChildElements(), model);
        }
        if (Util.isFunction(this.delegate.postRender)) {
            this.delegate.postRender(model, this.el);
        }
    };

    View.prototype.update = function(event) {
        if (Util.isFunction(this.delegate.update)) {
            if (this.delegate.update(event, this.el)) {
                this.dispose(event.model);
                this.render(event.model);
                return;
            }
        }
        for(var i = this.childViews.length - 1; i >= 0; i--) {
            if (!this.childViews[i].el.parentNode) {
                this.childViews[i].dispose(event.model);
                this.childViews.splice(i, 1);
            }
        }
        var placeholders = this.collectChildElements();
        for(var i = 0; i < this.childViews.length; i++) {
            this.childViews[i].update(event);
        }
        this.createChildViews(placeholders, event.model);
    };

    View.prototype.dispose = function(model) {
        for(var i = 0; i < this.childViews.length; i++) {
            var childDomElement = this.childViews[i].el;
            this.childViews[i].dispose(model);
        }
        if (Util.isFunction(this.delegate.dispose)) {
            this.delegate.dispose(model, this.el);
        }
        if (this.el) {
            while(this.el.firstChild) {
                this.el.removeChild(this.el.firstChild);
            }
        }
        this.childViews = [];
    };

    View.prototype.isChildViewElement = function(domElement) {
        for(var i = 0; i < this.childViews.length; i++) {
            if (this.childViews[i].el === domElement) {
                return true;
            }
        }
        return false;
    };

    View.prototype.containsElement = function(domElement) {
        if (domElement == this.el) {
            return false;
        }
        domElement = domElement.parentNode;
        while(domElement != null) {
            if (domElement === this.el) {
                return true;
            } else {
                for (var i = 0; i < this.childViews.length; i++) {
                    if (domElement === this.childViews[i].el) {
                        return false;
                    }
                }
            }
            domElement = domElement.parentNode;
        }
        return false;
    };

    exports.Controller = Controller;
    exports.View = View;
});