define('tmatesoft/engine-util', ['exports'], function(exports) {

    'use strict';

    var isObject = function(o) {
        return o && o.constructor == Object;
    };

    var isArray = function(o) {
        if (!Array.isArray) {
            return Object.prototype.toString.call(o) === '[object Array]';
        }
        return Array.isArray(o);
    };

    var isFunction = function(o) {
        return o && typeof o === 'function';
    };

    var getObjectKeySet = function(o, keys) {
        keys = keys || {};
        for (var key in o) {
            if (o.hasOwnProperty(key)) {
                keys[key] = true;
            }
        }
        return keys;
    }

    var getObjectKeyArray = function(o) {
        var keys = [];
        for (var key in o) {
            if (o.hasOwnProperty(key)) {
                keys.push(key);
            }
        }
        return keys;
    }

    var clone = function(o) {
        var c = o;
        if (isObject(o)) {
            c = {};
            for(var key in o) {
                if (o.hasOwnProperty(key)) {
                    c[key] = clone(o[key]);
                }
            }
        } else if (isArray(o)) {
            c = [];
            for(var i = 0; i < o.length; i++) {
                c.push(clone(o[i]));
            }
        }
        return c;
    };

    var extendClass = function(subclass, superclass) {
        subclass.prototype = new superclass();
        subclass.prototype.constructor = subclass;
    };

    var treesTraverse = function(trees, cb) {
        var queue = [];
        var paths = [];
        queue.push(trees);
        paths.push('');

        while(queue.length > 0) {
            var keys = {};
            var roots = queue.shift();
            var rootPath = paths.shift();
            var rootIds = getObjectKeySet(roots);
            for(var rootId in rootIds) {
                getObjectKeySet(roots[rootId], keys);
            }
            for(var key in keys) {
                var values = {};
                var empty = true;
                for(var rootId in rootIds) {
                    if (key in roots[rootId] && roots[rootId].hasOwnProperty(key)) {
                        values[rootId] = roots[rootId][key];
                        empty = false;
                    }
                }
                if (empty) {
                    continue;
                }
                var valuesPath = rootPath + key;
                var cbr = cb(key, roots, values, valuesPath);
                if (cbr === true) {
                    for(var rootId in rootIds) {
                        if (!isObject(values[rootId])) {
                            delete values[rootId];
                        }
                    }
                    queue.push(values);
                    paths.push(valuesPath + '.');
                } else if (cbr !== false) {
                    return cbr;
                }
            }
        }
        return true;
    };

    var pairEqual = function(f, s) {
        if (f === undefined || s === undefined || f === null || s === null) {
            return f === s;
        } else if (f.constructor.name !== s.constructor.name) {
            return false;
        } else if (isArray(f)) {
            if (f.length !== s.length) {
                return false;
            } else {
                for(var j = 0; j < f.length; j++) {
                    if (!pairEqual(f[j], s[j])) {
                        return false;
                    }
                }
            }
            return true;
        } else if (isObject(f)) {
            return treesEqual({ f : f, s : s });
        }
        return f === s;
    };

    var valuesEqual = function(valuesSet) {
        if (arguments.length > 1) {
            for(var i = 0; i < arguments.length - 1; i++) {
                if (!pairEqual(arguments[i], arguments[i + 1])) {
                    return false;
                }
            }
            return true;
        }

        var prev = undefined;
        for (var key in valuesSet) {
            if (!valuesSet.hasOwnProperty(key)) {
                continue;
            }
            if (prev !== undefined) {
                if (!pairEqual(prev, valuesSet[key])) {
                    return false;
                }
            }
            prev = valuesSet[key];
        }
        return true;
    };

    var treesEqual = function(trees) {
        var treeIds = getObjectKeySet(trees);
        var result = treesTraverse(trees, function(key, nodes, values) {
            var hasValues = false;
            for(var treeId in treeIds) {
                if (values[treeId] === undefined) {
                    return false;
                } else if (!isObject(values[treeId])) {
                    hasValues = true;
                }
            }
            return !hasValues || valuesEqual(values) ? true : values;
        });
        return result === true;
    };

    var iterateToPath = function(o, path, isSet) {
        if (!o || !path || path === null || path.length === 0) {
            return undefined;
        }
        var segments = path.split('.');
        var current = o;
        for(var i = 0; i < segments.length - 1; i++) {
            var key = segments[i];
            var couldTraverse = current.hasOwnProperty(key) && isObject(current[key]);
            if (isSet && !couldTraverse) {
                current[key] = {};
            } else if (!couldTraverse) {
                return undefined;
            }
            current = current[key];
        }
        var key = segments[segments.length - 1];
        return { node : current, key : key};
    };

    var setValueAtPath = function(o, path, value) {
        var pathNode = iterateToPath(o, path, true);
        if (pathNode) {
            pathNode.node[pathNode.key] = value;
        }
    };

    var getValueAtPath = function(o, path) {
        if (!path || path === '') {
            return o;
        }
        var pathNode = iterateToPath(o, path, false);
        if (pathNode) {
            return pathNode.node[pathNode.key];
        }
        return undefined;
    };

    function FlatObject() {
        this.map = {};
    };

    FlatObject.prototype.get = function(path) {
        var obj = undefined;
        for(var key in this.map) {
            if (!this.map.hasOwnProperty(key)) {
                continue;
            }
            if (key === path) {
                return this.map[key];
            } else if (path.indexOf(key + '.') === 0) {
                var value = this.map[key];
                if (isObject(value)) {
                    return getValueAtPath(value, path.substring(key.length + '.'.length));
                }
                return undefined;
            } else if (key.indexOf(path + '.') === 0) {
                obj = obj || {};
                setValueAtPath(obj, key.substring((path + '.').length), this.map[key]);
            }
        }
        return obj;
    };

    FlatObject.prototype.has = function(path) {
        for(var key in this.map) {
            if (!this.map.hasOwnProperty(key)) {
                continue;
            } else if (key === path || path.indexOf(key + '.') === 0 || key.indexOf(path + '.') === 0) {
                return true;
            }
        }
        return false;
    };

    FlatObject.prototype.isEmpty = function() {
        for(var key in this.map) {
            if (this.map.hasOwnProperty(key)) {
                return false;
            }
        }
        return true;
    };

    FlatObject.prototype.keys = function() {
        return getObjectKeyArray(this.map);
    };

    FlatObject.prototype.remove = function(path, excludePath) {
        var keys = getObjectKeySet(this.map);
        // remove value at path and all below it.
        for(var key in keys) {
            if (keys.hasOwnProperty(key)
                && ((!excludePath && key === path) || key.indexOf(path + '.') === 0)) {
                delete this.map[key];
            }
        }
    };

    FlatObject.prototype.set = function(path, value) {
        this.remove(path, true);
        for(var key in this.map) {
            if (!this.map.hasOwnProperty(key)) {
                continue;
            } else if (path.indexOf(key + '.') === 0) {
                var object = this.map[key];
                if (isObject(object)) {
                    setValueAtPath(object, path.substring((key + '.').length), value);
                    return;
                }
            }
        }
        this.map[path] = value;
    };

    exports.isObject = isObject;
    exports.isArray = isArray;
    exports.isFunction = isFunction;
    exports.clone = clone;
    exports.extendClass = extendClass;
    exports.getObjectKeySet = getObjectKeySet;
    exports.getObjectKeys = getObjectKeyArray;

    exports.setValueAtPath = setValueAtPath;
    exports.getValueAtPath = getValueAtPath;
    exports.FlatObject = FlatObject;

    exports.treesTraverse = treesTraverse;
    exports.valuesEqual = valuesEqual;

});